package ru.yandex.tours.storage.direction

import java.time.Month

import akka.actor.ActorSystem
import org.joda.time.LocalDate
import ru.yandex.tours.avia.AviaCacheClient
import ru.yandex.tours.model.Prices.DirectionBestPrice
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.model.search.{SearchType, HotelSearchRequest}
import ru.yandex.tours.model.search.SearchType.SearchType
import ru.yandex.tours.storage.minprice.MinPriceDao
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.concurrent.{AsyncBurstScaler, AsyncWorkQueue}
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.lang.Futures

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Random, Success}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.07.15
 */
class DirectionPriceStorage(dao: DirectionPriceDao,
                            minPriceDao: MinPriceDao,
                            aviaCacheClient: Option[AviaCacheClient],
                            searchType: SearchType)
                           (implicit akkaSystem: ActorSystem, ec: ExecutionContext) extends Logging {

  private val ZERO_AT = 15.minutes
  private val ONE_AT = 60.minutes

  private val NIGHTS_TO_SCAN = 1 to 20

  private case class Key(from: Int, ages: Seq[Int], month: Month, geoId: Int)
  private val throttler = new AsyncBurstScaler[Key](20.minutes)
  private val queue = new AsyncWorkQueue(1)

  private def shouldReload(entity: DirectionMonthBestPrice) = {
    val elapsed = ((System.currentTimeMillis() - entity.created) max 0).millis
    val p = ((elapsed - ZERO_AT) max Duration.Zero) / (ONE_AT - ZERO_AT)
    Random.nextDouble() < p
  }

  private def reload(from: Int, ages: Seq[Int], month: Month, geoId: Int): Future[Unit] = {
    log.info(s"Reloading direction min_price: from = $from, to = $geoId, ages = $ages, month = $month")
    val today = LocalDate.now
    val whenStart = today.nearMonthStart(month) max today
    val whenEnd = whenStart.atMonthEnd
    val realFrom = if (searchType == SearchType.ROOMS) 0 else from
    val priceFutures = minPriceDao.getPrices(realFrom, geoId, NIGHTS_TO_SCAN, ages, whenStart, whenEnd)
//    val futures = for (nights <- NIGHTS_TO_SCAN) yield {
//      val whenStart = today.nearMonthStart(month) max today
//      val whenEnd = whenStart.atMonthEnd
//      val realFrom = if (searchType == SearchType.ROOMS) 0 else from
//      minPriceDao.getPrices(realFrom, geoId, nights, ages, whenStart, whenEnd)
//    }

    def directionBestPrices(allPrices: Seq[MinPriceDao.PriceEntity]) = {
      for ((nights, prices) <- allPrices.filter(_.when > today).filter(_.price > 0).groupBy(_.nights)) yield {
        val minPrice = prices.minBy(_.price)
        DirectionBestPrice.newBuilder()
          .setSearchRequest(HotelSearchRequest(from, geoId, nights, minPrice.when, ages).toProto)
          .setPrice(minPrice.price)
          .build
      }
    }

    def withAviaPrice(bestPrice: DirectionBestPrice): Future[DirectionBestPrice] = {
      if (aviaCacheClient.isEmpty) return Future.successful(bestPrice)
      aviaCacheClient.get.getMinPrice(HotelSearchRequest(bestPrice.getSearchRequest)).map {
        case Some(aviaPrice) =>
          bestPrice.toBuilder
            .setPrice(bestPrice.getPrice + aviaPrice)
            .setWithFlight(true)
            .setFlightPrice(aviaPrice)
            .setRoomPrice(bestPrice.getPrice)
            .build
        case None =>
          bestPrice.toBuilder
            .setWithFlight(false)
            .build
      }
    }

    for {
      allPrices <- priceFutures
      directionPrices = directionBestPrices(allPrices.toSeq)
      withAvia <- Futures.lazyTraverse(directionPrices)(withAviaPrice)
      _ <- dao.update(from, ages, month, geoId, DirectionMonthBestPrice(geoId, month, withAvia.toVector))
    } yield {
      log.info(s"Reloaded direction min_price: from = $from, to = $geoId, ages = $ages, month = $month")
    }
  }

  def getPrices(from: Int, ages: Seq[Int], geoId: Int, months: Set[Month]): Future[Seq[DirectionMonthBestPrice]] = {
    dao.get(from, ages, geoId, months).andThen {
      case Success(entities) =>
        val map = entities.map(e => e.month -> e).toMap
        for (month <- months) {
          if (!map.contains(month) || shouldReload(map(month))) {
            throttler.throttle(Key(from, ages, month, geoId)) {
              queue.submit({
                reload(from, ages, month, geoId)
              }, deadline = 1.minute)
            }
          }
        }
    }
  }
}
