package ru.yandex.tours.storage.direction

import java.time.Month
import java.util

import akka.actor.ActorSystem
import org.apache.commons.codec.digest.DigestUtils
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.concurrent.AsyncWorkQueue

import scala.concurrent.{ExecutionContext, Future}

class LRUCacheDirectionPriceDao(dao: DirectionPriceDao, capacity: Int)
                               (implicit akkaSystem: ActorSystem,
                                ec: ExecutionContext) extends DirectionPriceDao with Logging {

  private val cache: util.Map[String, DirectionMonthBestPrice] = {
    new util.LinkedHashMap[String, DirectionMonthBestPrice](capacity, 0.75F, true) {
      private val cacheCapacity = capacity

      override def removeEldestEntry(eldest: util.Map.Entry[String, DirectionMonthBestPrice]): Boolean = {
        this.size() > this.cacheCapacity
      }
    }
  }

  private val queue = new AsyncWorkQueue(1)

  private def getKey(from: Int, ages: Seq[Int], month: Month, geoId: Int): String = {
    val key = s"$from:${ages.mkString(",")}:${month.getValue}:$geoId"
    DigestUtils.md5Hex(key)
  }

  private def getBackground(from: Int, ages: Seq[Int], geoId: Int, months: Set[Month]): Future[Unit] = {
    log.debug(s"Direction $geoId is not in LRU cache, querying from underlying storage")
    for (seq <- dao.get(from, ages, geoId, months)) yield {
      for (dmbp <- seq) yield {
        log.debug(s"Direction $geoId is fetched from underlying storage")
        cache.synchronized { cache.put(getKey(from, ages, dmbp.month, geoId), dmbp) }
      }
    }
  }

  override def update(from: Int, ages: Seq[Int], month: Month, geoId: Int,
                      entity: DirectionMonthBestPrice): Future[Unit] = {
    cache.put(getKey(from, ages, month, geoId), entity)
    dao.update(from, ages, month, geoId, entity)
  }


  override def get(from: Int, ages: Seq[Int], geoId: Int, months: Set[Month]): Future[Seq[DirectionMonthBestPrice]] = {
    cache.synchronized {
      val keysNotFound = months.filter(m => cache.get(getKey(from, ages, m, geoId)) == null)
      if (keysNotFound.nonEmpty) {
        queue.submit(
          {
            getBackground(from, ages, geoId, keysNotFound)
          })
      }
      val q = for {
        m <- months
        dmbp = cache.get(getKey(from, ages, m, geoId))
        if dmbp != null
      } yield dmbp
      Future.successful(q.toSeq)
    }
  }
}
