package ru.yandex.tours.storage.direction
import java.nio.ByteBuffer
import java.time.Month

import org.apache.commons.io.output.ByteArrayOutputStream
import ru.yandex.tours.model.Prices.DirectionBestPrice
import ru.yandex.tours.storage.yt.entities.DirectionPrice
import ru.yandex.tours.util.{Logging, ProtoIO}
import ru.yandex.travel.yt.{Factory, YtDao}
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.travel.yt.queries.{Compare, In}

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._
import scala.util.Try

class YtDirectionPriceDao(dao: YtDao[DirectionPrice])
                         (implicit ec: ExecutionContext) extends DirectionPriceDao with Logging {

  private def normalizeAges(ages: Seq[Int]): String = {
    ages.toVector.sorted.mkString(",")
  }

  private def marshallEntity(entity: DirectionMonthBestPrice): Array[Byte] = {
    val out = new ByteArrayOutputStream()
    entity.prices.foreach(_.writeDelimitedTo(out))
    out.toByteArray
  }

  override def update(from: Int, ages: Seq[Int], month: Month,
                      geoId: Int, entity: DirectionMonthBestPrice): Future[Unit] = {
//    log.debug("Saving DirectionPrice for " + geoId.toString)
    val item = new DirectionPrice(from, normalizeAges(ages), month.getValue, geoId, marshallEntity(entity))
    val future = dao.put(item).asScalaFuture.toUnit
    future.onFailure {
      case t => log.error("An error has occured on update: " + t.getMessage)
    }
    future
  }

  override def get(from: Int, ages: Seq[Int], geoId: Int, months: Set[Month]): Future[Seq[DirectionMonthBestPrice]] = {
    val started = System.currentTimeMillis()
    log.debug("Starting YT query")
    val query = dao.select(
      Compare.eq("From", from),
      Compare.eq("Ages", normalizeAges(ages)),
      Compare.eq("GeoId", geoId),
      new In("Month", seqAsJavaList(months.map(m => m.getValue).toSeq))).asScalaFuture
    query.onSuccess {
      case _ => log.debug("Done YT query in " + (System.currentTimeMillis() - started).toString)
    }
    query.onFailure {
      case t => log.error("An error has occured on get: " + t.getMessage)
    }

    for (priceList <- query) yield {
      for (price <- priceList.asScala) yield  {
        val prices = Try(ProtoIO.loadFromBytes(price.getBestPrice, DirectionBestPrice.PARSER).toVector)
          .getOrElse(Vector.empty)
        DirectionMonthBestPrice(geoId, Month.of(price.getMonth), prices, price.getWriteTime)
      }
    }
  }
}
