package ru.yandex.tours.storage.minprice

import java.time.Month

import org.joda.time.{Days, LocalDate}
import ru.yandex.tours.model.Prices.{DirectionBestPrice, PriceGraph}
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.util.proto._
import ru.yandex.tours.search.Defaults
import ru.yandex.tours.services.MinPriceService
import ru.yandex.tours.storage.direction.{DirectionPriceDao, DirectionPriceStorage}
import ru.yandex.tours.storage.minprice.MinPriceDao.PriceEntity
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.lang.Dates._

import scala.collection.JavaConversions._
import scala.concurrent.{ExecutionContext, Future}

class LocalMinPriceService(minPriceStorage: MinPriceStorage,
                           directionPriceDao: DirectionPriceDao)
                          (implicit ec: ExecutionContext) extends MinPriceService {


  private val MONTHS_TO_SCAN = 3

  override def getMinPriceGraph(request: HotelSearchRequest, graphLength: Int): Future[PriceGraph] = {
    getGraphForNights(request.from, request.to, request.when, request.nights, request.agesSerializable, graphLength)
  }

  override def getMinPriceGraph(from: Int, to: Int, when: LocalDate, nights: Int, ages: Seq[Int],
                                graphStart: LocalDate, graphEnd: LocalDate): Future[PriceGraph] = {
    for {
      prices <- minPriceStorage.getPrices(from, to, nights, ages, graphStart, graphEnd)
    } yield {
      buildGraph(from, to, when, nights, ages, graphStart, prices, Days.daysBetween(graphStart, graphEnd).getDays + 1)
    }
  }

  override def getDirectionMinPrice(from: Int, to: Int): Future[Option[DirectionBestPrice]] = {
    val currentMonth = Month.of(LocalDate.now.getMonthOfYear)
    val defaultMonths = ((0 until MONTHS_TO_SCAN) map (currentMonth.plus(_))).toSet
    directionPriceDao.get(from, Defaults.TWO_ADULTS, to, defaultMonths).map { prices ⇒
      prices.flatMap(_.prices).minOptBy(_.getPrice)
    }
  }

  private def getGraphForNights(from: Int, to: Int, when: LocalDate, nights: Int,
                                ages: Seq[Int],
                                graphLength: Int): Future[PriceGraph] = {
    val graphStart = when.minusDays(graphLength - 1)
    val graphEnd = when.plusDays(graphLength - 1)
    for {
      prices <- minPriceStorage.getPrices(from, to, nights, ages, graphStart, graphEnd)
    } yield {
      val today = new LocalDate()
      val actualPrices = prices.filter(_.when >= today)
      val startDate = getStartDate(actualPrices, today, when, graphLength)
      val pricesForGraph = actualPrices.filter(isInSameGraph(startDate, graphLength))
      buildGraph(from, to, when, nights, ages, startDate, pricesForGraph, graphLength)
    }
  }

  /*
    Here we want to choose date, which will be the start of the graph
    Also, there should be some gap between "start date" / "end date" and "requested date",
    because we don't want "requested date" to be at to the end of the graph.
    But if "today" is near to "requested date", start date should be "today"
    P.S
      "start date" - start of the graph
      "end date" - end of the graph
      "requested date" - "when" date from request
   */
  private def getStartDate(actualPrices: Iterable[PriceEntity], today: LocalDate, when: LocalDate, graphLength: Int) = {
    val gapLength = graphLength / 4
    if (Days.daysBetween(today, when).getDays <= graphLength - gapLength) {
      today
    } else {
      val maxPriceCountShift = (-(graphLength - gapLength) to -gapLength).maxBy(dayShift => {
        val startDay = when.plusDays(dayShift)
        actualPrices.count(isInSameGraph(startDay, graphLength))
      })
      when.plusDays(maxPriceCountShift)
    }
  }

  /*
    Checks whether @otherPrice would be in the graph with starting at @start with length @graphLength
   */
  private def isInSameGraph(start: LocalDate, graphLength: Int)(otherPrice: PriceEntity) = {
    val when = otherPrice.when
    (when >= start) && Days.daysBetween(start, when).getDays.abs < graphLength
  }

  private def mergeMinPrices(prices: Seq[PriceEntity], operators: Set[Int]): Int = {
    if (prices.isEmpty) MinPriceService.UNKNOWN_PRICE
    else {
      val withPrices = prices.filter(_.price > 0)
      if (withPrices.isEmpty) {
        if (prices.map(_.operatorId).toSet == operators) MinPriceService.NO_TOURS
        else MinPriceService.UNKNOWN_PRICE
      } else withPrices.map(_.price).min
    }
  }

  private def buildOperatorPrice(prices: Seq[PriceEntity], operators: Set[Int]) = {
    val builder = PriceGraph.PriceEntity.newBuilder()
    for (price <- prices) yield {
      builder.addOperatorPriceBuilder()
        .setOperatorId(price.operatorId)
        .setPrice(price.price)
    }
    builder.setMinPrice(mergeMinPrices(prices, operators)).build()
  }

  private def buildGraph(from: Int, to: Int, when: LocalDate, nights: Int, ages: Seq[Int],
                         startDate: LocalDate,
                         prices: Iterable[PriceEntity],
                         graphLength: Int) = {
    require(prices.forall(_.when >= startDate), s"Expected startDate be <= to all entities in graph")
    val dayToPrice = prices.map(t => Days.daysBetween(startDate, t.when).getDays.abs -> t).toMultiMap
      .withDefaultValue(Seq.empty)

    val operators = prices.map(p => p.operatorId).toSet
    val priceArray = (0 until graphLength).map(dayToPrice).map(mergeMinPrices(_, operators)).map(Int.box)
    val operatorPriceArray = (0 until graphLength).map(dayToPrice).map(buildOperatorPrice(_, operators))
    PriceGraph.newBuilder()
      .addAllOBSOLETEPrice(asJavaIterable(priceArray))
      .addAllPriceEntity(asJavaIterable(operatorPriceArray))
      .setIndex(Days.daysBetween(startDate, when).getDays.abs)
      .setWhen(fromLocalDate(when))
      .addAllAge(ages.map(Int.box))
      .setNights(nights)
      .setFrom(from)
      .setTo(to)
      .build()
  }
}
