package ru.yandex.tours.storage.minprice

import org.joda.time.LocalDate
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchProducts.HotelSnippet
import ru.yandex.tours.model.search.SearchResults.HotelSearchResult
import ru.yandex.tours.services.MinPriceService
import ru.yandex.tours.storage.minprice.MinPriceDao.PriceEntity
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.lang.Dates._

import scala.collection.JavaConversions._
import scala.concurrent.{ExecutionContext, Future}

class MinPriceStorage(dao: MinPriceDao)(implicit ec: ExecutionContext) extends Logging {

  def getPrices(from: Int,
                to: Int,
                nights: Int,
                ages: Seq[Int],
                whenStart: LocalDate,
                whenEnd: LocalDate): Future[Iterable[PriceEntity]] = {
    dao.getPrices(from, to, nights, ages, whenStart, whenEnd)
  }

  def update(request: HotelSearchRequest, searchResult: HotelSearchResult): Future[Unit] = {
    if (searchResult.getProgress.getIsFinished) {
      val completed = SimpleBitSet.from(searchResult.getProgress.getOperatorCompleteSet).toSet
      val failed = SimpleBitSet.from(searchResult.getProgress.getOperatorFailedSet).toSet
      val successfulOperators = completed diff failed
      if (searchResult.getHotelSnippetCount == 0) {
        updateEmptyResult(request, successfulOperators)
      } else {
        updateNonEmptyResult(request, successfulOperators, failed, searchResult.getHotelSnippetList)
      }
    } else {
      Future.successful(())
    }
  }

  private def emptyPrices(request: HotelSearchRequest, operator: Int) = {
    for {
      nights <- request.nightsRange
      date <- request.dateRange
    } yield PriceEntity(request.from, request.to, nights, request.agesSerializable, date, operator, MinPriceService.NO_TOURS)
  }

  private def nonEmptyPrices(request: HotelSearchRequest, operator: Int, snippets: Iterable[HotelSnippet]) = {
    snippets.groupBy(s => (s.getDateMin, s.getNightsMin)).map {
      case ((when, nights), snips) =>
        val minPriceSnippet = snips.minBy(_.getPriceMin)
        PriceEntity(request.from, request.to, nights, request.agesSerializable, when.toLocalDate, operator, minPriceSnippet.getPriceMin)
    }
  }

  private def updateEmptyResult(request: HotelSearchRequest, operators: Iterable[Int]) = {
    val prices = operators.flatMap(operator => emptyPrices(request, operator))
    val result = dao.update(prices)
    result onFailure {
      case e => log.error("Can not update min prices!", e)
    }
    result
  }

  private def updateNonEmptyResult(request: HotelSearchRequest,
                                   operators: Set[Int],
                                   failedOperators: Set[Int],
                                   snippets: Iterable[HotelSnippet]) = {
    // Some moot point. We can not say anything about min price if we have flex snippets...
    if (snippets.forall(isNotFlex)) {
      val operatorSnippets = (for {
        snippet <- snippets
        source <- snippet.getSourceList
      } yield source.getOperatorId -> snippet).toMultiMap

      val prices = for (operator <- operators.toSeq if !failedOperators.contains(operator)) yield {
        val snippets = operatorSnippets.getOrElse(operator, List.empty)
        if (snippets.isEmpty) emptyPrices(request, operator)
        else nonEmptyPrices(request, operator, snippets)
      }
      val result = dao.update(prices.flatten)
      result onFailure {
        case e => log.error("Can not update min prices!", e)
      }
      result
    } else {
      log.info(s"Ignored flex snippets for request $request")
      Future.successful(())
    }
  }

  private def isNotFlex(snippet: HotelSnippet) =
    snippet.getDateMin == snippet.getDateMax &&
      snippet.getNightsMin == snippet.getNightsMax &&
      snippet.getSourceCount == 1
}
