package ru.yandex.tours.storage.minprice

import org.joda.time.LocalDate
import ru.yandex.tours.storage.minprice.MinPriceDao.PriceEntity
import ru.yandex.tours.storage.yt.entities.Price
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.lang.Futures
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.travel.yt.YtDao
import ru.yandex.travel.yt.queries.{Compare, In, QueryPart}

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}

class YtMinPriceDao(dao: YtDao[Price])
                   (implicit ec: ExecutionContext) extends MinPriceDao {


  private def normalizeAges(ages: Seq[Int]): String = {
    ages.toVector.sorted.mkString(",")
  }

  override def update(prices: Iterable[MinPriceDao.PriceEntity]): Future[Unit] = {
    val futures = Seq.newBuilder[Future[Unit]]
    val priceItems = prices.map(price => new Price(
      price.from,
      price.to,
      price.nights,
      normalizeAges(price.ages),
      price.when.toMillis,
      price.operatorId,
      price.price
    )).toList.asJava
    dao.put(priceItems).asScalaFuture.toUnit
  }

  override def getPrices(from: Int, to: Int, nights: Int, ages: Seq[Int],
                         whenStart: LocalDate, whenEnd: LocalDate): Future[Iterable[PriceEntity]] = {
    val query = dao.select(
      Compare.eq("From", from),
      Compare.eq("To", to),
      Compare.eq("Nights", nights),
      Compare.eq("Ages", normalizeAges(ages)),
      Compare.ge("When", whenStart.toMillis),
      Compare.le("When", whenEnd.toMillis)).asScalaFuture
    for (priceList <- query) yield {
      for (price <- priceList.asScala) yield {
        PriceEntity(price.getFrom, price.getTo, price.getNights, price.getAges.split(",").map(_.toInt),
          price.getWhen.toLocalDate, price.getOperator, price.getMinPrice)
      }
    }
  }

  override def getPrices(from: Int, to: Int, nights: Seq[Int], ages: Seq[Int],
                         whenStart: LocalDate, whenEnd: LocalDate): Future[Iterable[PriceEntity]] = {
    val query = dao.select(
      Compare.eq("From", from),
      Compare.eq("To", to),
      new In("Nights", nights.toList.asJava),
      Compare.eq("Ages", normalizeAges(ages)),
      Compare.ge("When", whenStart.toMillis),
      Compare.le("When", whenEnd.toMillis)).asScalaFuture
    for (priceList <- query) yield {
      for (price <- priceList.asScala) yield {
        PriceEntity(price.getFrom, price.getTo, price.getNights, price.getAges.split(",").map(_.toInt),
          price.getWhen.toLocalDate, price.getOperator, price.getMinPrice)
      }
    }
  }

}
