package ru.yandex.tours.storage.subscriptions

import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicInteger

import akka.util.Timeout
import play.api.libs.json.{JsObject, JsValue, Json}
import ru.yandex.tours.model.subscriptions.Subscriptions.State
import ru.yandex.tours.model.subscriptions.{Subscription, Uid, UserIdentity, YandexUid}
import ru.yandex.tours.subscriptions.model.ModelConverters._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.{Futures, SideEffectTry}
import ru.yandex.tours.util.spray._
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 21.08.15
 */
class HttpSubscriptionStorage(httpClient: AsyncHttpClient, host: String, port: Int)
                             (implicit ec: ExecutionContext, timeout: Timeout)
  extends SubscriptionStorage with Logging {

  import HttpSubscriptionStorage._

  private val baseUri = Uri(s"http://$host:$port/api/1.x/subscriptions/service/tours")

  private val headers = List("Accept" -> "application/json")

  private lazy val allTokens: Seq[String] = Await.result({
    val uri = baseUri / "token"
    httpClient.get(uri).map {
      case (StatusCodes.OK, json) =>
        Json.parse(json).as[Seq[String]]
      case (status, _) => sys.error("Unexpected status code: " + status)
    }
  }, timeout.duration)

  override def getSubscription(user: UserIdentity, id: String): Future[Option[Subscription]] = {
    httpClient.get(baseUri / "user" / user.identityType / user.identity / id, headers).map {
      case (StatusCodes.OK, json) => parseSubscription(Json.parse(json)).toOption
      case (StatusCodes.NotFound, _) => Option.empty
      case (status, _) => sys.error("Unexpected status code: " + status)
    }
  }

  override def getSubscriptions(user: UserIdentity): Future[Seq[Subscription]] = {
    val failed = new AtomicInteger(0)
    httpClient.get(baseUri / "user" / user.identityType / user.identity, headers).map {
      case (StatusCodes.OK, json) =>
        val array = Json.parse(json).as[Seq[JsValue]]
        for {
          elem <- array
          subscription <- parseSubscription(elem).onFailure(_ => failed.incrementAndGet()).toOption
        } yield subscription
      case (status, _) => sys.error("Unexpected status code: " + status)
    }.andThen { case _ if failed.get > 0 =>
      log.warn(s"Failed to parse ${failed.get} subscriptions")
    }
  }

  override def getAllSubscriptions: Future[Seq[Subscription]] = {
    val failed = new AtomicInteger(0)
    def getTokenSubscriptions(token: String): Future[Seq[Subscription]] = {
      httpClient.get(baseUri / "token" / token, headers).map {
        case (StatusCodes.OK, json) =>
          val array = Json.parse(json).as[Seq[JsValue]]
          for {
            elem <- array
            subscription <- parseInternalSubscription(elem).onFailure(_ => failed.incrementAndGet()).toOption
          } yield subscription
        case (status, _) => sys.error("Unexpected status code: " + status)
      }
    }

    Futures.lazySequence {
      for (token <- allTokens.iterator) yield {
        getTokenSubscriptions(token)
      }
    }.map(_.flatten).andThen { case _ if failed.get > 0 =>
      log.warn(s"Failed to parse ${failed.get} subscriptions")
    }
  }

  override def disableSubscription(user: UserIdentity, id: String): Future[Unit] = {
    httpClient.delete(baseUri / "user" / user.identityType / user.identity / id).map {
      case (StatusCodes.OK, _) => ()
      case (StatusCodes.NotFound, _) => ()
      case (status, _) => sys.error("Unexpected status code: " + status)
    }
  }
}

object HttpSubscriptionStorage {

  def parseSubscription(json: JsValue): Try[Subscription] = Try {
    val userJson = (json \ "user").as[JsObject]
    val user =
      if (userJson.value.contains("uid")) Uid(userJson.value("uid").as[String].toLong)
      else YandexUid(userJson.value("yandexuid").as[String])
    val id = (json \ "id").as[String]
    val email = (json \ "delivery" \ "email" \ "address").as[String]
    val period = (json \ "delivery" \ "email" \ "period").as[Int]
    val state = (json \ "state" \ "value").as[String]
    val query = (json \ "request" \ "http_query").as[String]

    Subscription(
      user,
      id,
      email,
      FiniteDuration(period, TimeUnit.MINUTES),
      State.valueOf(state),
      Uri.Query(query).toSubject
    )
  }
  
  def parseInternalSubscription(json: JsValue): Try[Subscription] = Try {
    val userJson = (json \ "user").as[JsObject]
    val user =
      if (userJson.value.contains("uid")) Uid(userJson.value("uid").as[String].toLong)
      else YandexUid(userJson.value("yandexuid").as[String])
    val id = (json \ "id").as[String]
    val email = (json \ "delivery" \ "email" \ "address").as[String]
    val period = (json \ "delivery" \ "email" \ "period" \ "length").as[Int]
    val unit = (json \ "delivery" \ "email" \ "period" \ "time_unit").as[String]
    val state = (json \ "state" \ "value").as[String]
    val query = (json \ "request" \ "source" \ "http_query").as[String]

    Subscription(
      user,
      id,
      email,
      FiniteDuration(period, TimeUnit.valueOf(unit)),
      State.valueOf(state),
      Uri.Query(query).toSubject
    )
  }
}