package ru.yandex.tours.tools

import java.io.{File, FileInputStream}

import akka.actor.ActorSystem
import com.typesafe.config.ConfigFactory
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.db.geomapping.{Add, GeoMappingTables}
import ru.yandex.tours.db.{DBWrapper, Transaction, Transactions}
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import slick.driver.MySQLDriver.api._

import scala.concurrent.{Future, Await}
import scala.concurrent.duration.Duration
import scala.io.Source
import scala.util.control.NonFatal

/**
  * Created by asoboll on 03.12.15.
  */
object GeoMappingsToMysql extends App {
  val conf = ConfigFactory.load("application.local.conf")
  val as = ActorSystem("geo-mapping-tool", conf)
  implicit val ec = as.dispatcher
  lazy val db = new DBWrapper(
    Database.forConfig("tours.mysql", conf)
  )

  val dataTypesToCopy = Seq(
    DataTypes.departures,
    DataTypes.countries,
    DataTypes.cities,
    DataTypes.airports
  )

  dataTypesToCopy.foreach(dataType => DeleteTable(GeoMappingTables.getQuery(dataType)))
  dataTypesToCopy.foreach(dataType => CreateTable(GeoMappingTables.getQuery(dataType)))
  Transactions.withTransaction(db) { transaction =>
    Future.sequence(dataTypesToCopy.map(dataType => writeFromFile(dataType, transaction)))
  }

  private def DeleteTable[T <:Table[_]](table:TableQuery[T]) = {
    try {
      Await.result(db.run(table.schema.drop), Duration.Inf)
    } catch {
      case NonFatal(t) => None
    }
  }

  private def CreateTable[T <:Table[_]](table:TableQuery[T]) = {
    Await.result(db.run(table.schema.create), Duration.Inf)
  }

  private def writeFromFile(dataType:DataType, transaction: Transaction) = {
    val is = new FileInputStream(new File("tours-data/data/" + dataType.getName))
    val mapping = Source.fromInputStream(is).getLines().map {
      case Tabbed(IntValue(partnerId), IntValue(geoId), pId) => (Add, GeoMappingShort(Partners(partnerId), geoId, pId))
    }
    GeoMappingTables.put(db, dataType, transaction, mapping.toIterable)
  }
}