package ru.yandex.tours.tools

import org.apache.commons.codec.binary.Base64
import org.apache.hadoop.hbase.HBaseConfiguration
import org.apache.hadoop.hbase.client._
import org.apache.hadoop.hbase.util.Bytes
import org.joda.time.format.DateTimeFormat
import org.joda.time.{DateTime, DateTimeZone, Minutes}
import ru.yandex.tours.util.{GZip, Logging}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.05.15
 */
trait HBaseTool extends Tool with Logging {
  private val scanWindow = Minutes.minutes(5)
  private val keyFormat = DateTimeFormat.forPattern("yyyy-MM-dd@HH:mm:ss.SSS@")

  val familyName = Bytes.toBytes("event")

  val bruny = {
    val conf = HBaseConfiguration.create()
    conf.set("hbase.zookeeper.quorum", "w160h.hdp.yandex.net,w159h.hdp.yandex.net,w168h.hdp.yandex.net")
    conf.set("zookeeper.znode.parent", "/bruny/hbase")
    conf
  }

  val klarke = {
    val conf = HBaseConfiguration.create()
    conf.set("hbase.zookeeper.quorum", "w001h.hdp.yandex.net,w002h.hdp.yandex.net,w003h.hdp.yandex.net,w004h.hdp.yandex.net,w005h.hdp.yandex.net")
    conf.set("zookeeper.znode.parent", "/klarke/hbase")
    conf
  }

  implicit class RichRow(result: Result) {
    def value(name: String) = result.getValue(familyName, Bytes.toBytes(name))
    def str(name: String) = Bytes.toString(value(name))
    def int(name: String) = Bytes.toString(value(name)).toInt
    def optStr(name: String) = Option(value(name)).map(Bytes.toString)
    def compressedBytes(name: String) = GZip.decompress(Base64.decodeBase64(value(name)))
    def compressedProto[T](name: String, parser: Array[Byte] => T) = parser(compressedBytes(name))
  }


  def getScanner(table: Table, from: DateTime, until: DateTime): ResultScanner = {
    val startKey = from.withZone(DateTimeZone.UTC).minus(scanWindow).toString(keyFormat)
    val endKey = until.withZone(DateTimeZone.UTC).toString(keyFormat)
    val minVersion = from.getMillis + 1
    val maxVersion = until.getMillis
    log.debug(s"Scanning from [$startKey] until [$endKey] versions [$minVersion, $maxVersion)")

    val scan = new Scan(Bytes.toBytes(startKey), Bytes.toBytes(endKey))
    scan.setTimeRange(minVersion, maxVersion)
    scan.setCaching(5000)
    scan.setCacheBlocks(false)

    val scanner = table.getScanner(scan)
    scanner
  }
}
