package ru.yandex.tours.tools

import ru.yandex.tours.app.{AkkaSupport, Application, DefaultEnvironment}
import ru.yandex.tours.db.geomapping.{GeoMappingRecordProcessor, GeoMappingUtils}
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.geo.matching.Hypothesis
import ru.yandex.tours.geo.partners.PartnerRegion
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.Tabbed

import scala.concurrent.Await
import scala.concurrent.duration.Duration

/**
  * Created by asoboll on 25.07.16.
  */
object MappingsExport extends Application with DefaultEnvironment with AkkaSupport with ToolDataHolders with TestDb {

  val db = prodDb
  val dataTypes = Seq(DataTypes.countries, DataTypes.cities)
  val partner = Partners.lt
  val partners = Seq(partner)

  val partnerTree = partnerTrees(partner)

  lazy val hypotheses = geoMatchingHypotheses(partner)

  def mappings: Iterable[GeoMappingShort] = Await.result(
    GeoMappingUtils.getMappings(db, dataTypes, partners).map { recordsMap =>
      GeoMappingRecordProcessor.getMapping(recordsMap.values.flatten.toSeq)
    }, Duration.Inf)

  def printMappings(partnerMappings: Iterable[GeoMappingShort]) = {
    IO.printFile("ya_lt_geo_mappings.tsv") { pw =>
      pw.println("ya_id\tya_ru_name\tya_en_name\tya_country_id\tya_country_ru_name\tya_country_en_name\tlt_id\tlt_ru_name\tlt_en_name")
      //pw.println("geoId\tbookingId\truName\tenName")
      for {
        record <- partnerMappings
        region <- partnerTree.region(record.partnerId)
        yaRegion <- regionTree.region(record.geoId)
      } {
        val country = regionTree.country(yaRegion)
        pw.println(Tabbed(
          record.geoId,
          yaRegion.name.ruName,
          yaRegion.name(Languages.en),
          country.fold("")(_.id.toString),
          country.fold("")(_.name.ruName),
          country.fold("")(_.name(Languages.en)),
          record.partnerId,
          region.name.ruName,
          region.name(Languages.en)
        ))
        //pw.println(Tabbed(record.geoId, record.partnerId, region.name(Languages.ru), region.name(Languages.en)))
      }
    }
  }

  def notMatched = hypotheses.filter(partner, (yaid, prid) =>
    !geoMappingHolder.getPartnerDestination(partner, yaid).contains(prid)
  )

  def printHypotheses(hyps: Iterable[Hypothesis]) = {
    def greater(a: Hypothesis, b: Hypothesis): Boolean = {
      if (a.confidence > b.confidence * 1.5)
        return true
      false
    }

    IO.printFile("booking_hypotheses.tsv") { pw =>
      pw.println("geoId\tbookingId\truName\tenName\tconfidence\tbookingIdIfOverride\tbookingIdIfAlternative\tbookingName\tgeoIdIfOverride\tgeoIdIfAlternative\tgeoName\tlackOfConfidence")
      for (hyp <- hyps) {
        val byPartner = geoMatchingHypotheses.byPartnerRegion(hyp.partnerRegion).filterNot(_ == hyp).filterNot(greater(hyp, _)).headOption.fold("")(_)
        val byYandex = geoMatchingHypotheses.byYandexRegion(hyp.yandexRegion, partner).filterNot(_ == hyp).filterNot(greater(hyp, _)).headOption.fold("")(_)
        val partnerRegion = partnerTree.region(hyp.partnerId).fold("")(_)

        pw.println(Tabbed(
          hyp.yandexRegion.id,
          PartnerRegion.originalId(hyp.partnerId),
          partnerRegion(_.name(Languages.ru)),
          partnerRegion(_.name(Languages.en)),
          hyp.confidence,
          byYandex(ya => if (greater(ya, hyp)) ya.partnerId else ""),
          byYandex(ya => if (!greater(ya, hyp)) ya.partnerId else ""),
          byYandex(ya => partnerTree.region(ya.partnerId).fold("")(_.name(Languages.ru))),
          byPartner(p => if (greater(p, hyp )) p.yandexRegion.id.toString else ""),
          byPartner(p => if (!greater(p, hyp )) p.yandexRegion.id.toString else ""),
          byPartner(p => regionTree.region(p.yandexRegion.id).fold("")(_.getName)),
          if (hyp.confidence < 1.0) "lackOfConfidence" else ""
        ))
      }
    }
  }

  printMappings(mappings)

//  printHypotheses(notMatched)

}
