package ru.yandex.tours.tools

import java.io.{File, PrintWriter}
import java.util.concurrent.ArrayBlockingQueue
import java.util.concurrent.atomic.LongAdder

import akka.actor.ActorSystem
import akka.util.Timeout
import play.api.libs.json.Json
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.http.NingHttpClient
import ru.yandex.tours.util.parsing.Tabbed
import spray.http.StatusCodes

import scala.concurrent.duration._
import scala.concurrent.{Await, Future}
import scala.util.control.NonFatal

object MetaImageUpdaterTool extends App {
  val as = ActorSystem()
  implicit val ec = as.dispatcher
  implicit val to = Timeout(30.seconds)

  case class ImageForMeta(group: String, name: String)

  assert(args.length == 2, "2 arguments should be specified: " +
    "1. Path to images file. " +
    "2. Number of worker threads")

  val threadNumber = args(1).toInt
  val httpClient = new NingHttpClient(None)

  val queue = new ArrayBlockingQueue[ImageForMeta](1000)
  val downloaded = new LongAdder()
  val failed = new LongAdder()

  val successLog = new PrintWriter(new File(s"success.meta.tsv"))
  val failedLog = new PrintWriter(new File(s"failure.meta.tsv"))

  def getAllImages: Iterator[ImageForMeta] = {
    IO.readLines(args(0)).map {
      case Tabbed(group, name) => ImageForMeta(group, name)
    }
  }

  val total = getAllImages.size
  println("Total images: " + total)

  def deleteMeta(image: ImageForMeta): Future[Unit] = {
    httpClient.get(s"http://localhost:8801/delete-tours/${image.group}/${image.name}/meta").map {
      case (StatusCodes.OK, _) =>
      case (sc, entity) => sys.error(s"Not 200: $sc, $entity")
    }
  }

  def getHash(image: ImageForMeta): Future[String] = {
    httpClient.get(s"http://localhost:8801/getinfo-tours/${image.group}/${image.name}/meta").map {
      case (StatusCodes.OK, entity) =>
        (Json.parse(entity) \ "NNetFeatures" \ "fc_encoder_96").as[String]
      case (sc, entity) => sys.error(s"Not 200: $sc, $entity")
    }
  }

  def updateMeta(image: ImageForMeta): Unit = {
    try {
      val result = for {
        _ <- deleteMeta(image)
        hash <- getHash(image)
      } yield hash
      val hash = Await.result(
        result,
        30.seconds
      )
      downloaded.increment()
      successLog.println(Tabbed(
        image.group,
        image.name,
        hash
      ))
      successLog.flush()
    } catch {
      case NonFatal(e) =>
        println(s"Failed to download $image: ${e.getMessage}")
        failed.increment()
        failedLog.println(Tabbed(image.group, image.name))
        failedLog.flush()
    }
  }


  new Thread("pusher") {
    override def run(): Unit = {
      getAllImages.foreach(queue.put)
    }
  }.start()

  def starWorker(i: Int): Unit = {
    new Thread("worker-" + i) {
      override def run(): Unit = {
        do {
          val i = queue.take()
          updateMeta(i)
        } while (true)
      }
    }.start()
  }

  (0 until threadNumber).foreach(starWorker)

  val started = System.currentTimeMillis()
  new Thread("stats-logger") {
    override def run(): Unit = {
      while (true) {
        Thread.sleep(5000)
        val now = System.currentTimeMillis()
        val elapsed = (now - started).millis.toMinutes
        println(s"Downloaded ${downloaded.intValue()} images out of $total in $elapsed minutes. " +
          s"Failed: ${failed.intValue}")
      }
    }
  }.start()

}
