package ru.yandex.tours.tools

import java.io.{File, PrintWriter}

import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.tools.GeoMappingTool.Mapping
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.Tabbed

import scala.io.Source

object PartnerMatchingConverter extends Tool {
  val synonymIds = Map(
    10994 -> List(1000000145, 1000000167, 1000000144)
  )

  val customMappings = Seq(
    (3, 116386, "116386"),          // HOTELS-1638
    (3, 1000000171, "1000000171"),  // HOTELS-1638
    (3, 1000000172, "1000000172"),  // HOTELS-1638
    (3, 1000000173, "1000000173"),  // HOTELS-1638
    (2, 1, "r4224")                 // московская область букинга
  )

  private def convert(from: Seq[String], to: String, getIds: Mapping => Seq[String], partner: Partner) {
    val partner2geoId = from.flatMap(f => GeoMappingTool.readGeoMapping(new File(f)))
    val pw = new PrintWriter(to)
    partner2geoId.toVector
      .groupBy(_.geoId).mapValues(_.reduce(_ + _))
      .values.toVector
      .sortBy(_.geoId.toString).foreach { x =>
      getIds(x) match {
        case Seq(partnerId) =>
          pw.println(partner.id + "\t" + x.geoId + "\t" + partnerId)
          synonymIds.getOrElse(x.geoId, Nil).foreach {
            geoId2 => pw.println(partner.id + "\t" + geoId2 + "\t" + partnerId)
          }
        case Seq() =>
        case _ => println(s"Duplicate $partner id for: ${x.geoId}: ${getIds(x)}")
      }
    }
    pw.close()
  }

  private def convertPartner(partner: Partner, getIds: Mapping => Seq[String]) = {
    convert(
      Seq(
        "util-heavy/python/geo/data/matchingresults/geo2AllPartners.resorts.dupsresolved.withbookingregions.withoutbigunpopular.tsv"
      ),
      s"tours-data/data/${partner}_cities.tsv",
      getIds,
      partner
    )

    convert(
      Seq(
        "util-heavy/python/geo/data/matchingresults/geo2AllPartners.countries.tsv"
      ),
      s"tours-data/data/${partner}_countries.tsv",
      getIds,
      partner
    )
  }

  def merge(partners: Partner*): Unit = {
    IO.printFile(s"tours-data/data/cities.tsv") { pw =>
      customMappings.foreach {
        case (partnerId, geoId, localId) =>
          pw.println(Tabbed(partnerId, geoId, localId))
      }
      partners.foreach { partner =>
        Source.fromFile(s"tours-data/data/${partner}_cities.tsv").getLines().foreach {
          line => pw.println(line)
        }
      }
    }
  }

  convertPartner(Partners.ostrovok, _.ostrovokId)
  convertPartner(Partners.sunmar, _.sunmarId)
  convertPartner(Partners.booking, _.bookingId)
  convertPartner(Partners.oktogo, _.oktogoId)
  convertPartner(Partners.hotels101, _.hotels101Id)
  convertPartner(Partners.lt, _.newLtId)
  convertPartner(Partners.sodis, _.sodisId)

  merge(
    Partners.lt,
    Partners.sunmar,
    Partners.hotels101,
    Partners.booking,
    Partners.oktogo,
    Partners.ostrovok,
    Partners.sodis
  )
}
