package ru.yandex.tours.tools

import java.io.{File, FileInputStream, PrintWriter}

import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}

import scala.collection.mutable

object SunmarMatchingConverter extends App {
  import GeoMappingTool.readGeoMapping

  val airports = new mutable.HashSet[String]
  val resorts = new mutable.HashSet[String]
  val beaches = new mutable.HashSet[String]

  val children = new mutable.HashMap[String, mutable.ArrayBuffer[String]]()

  var airport = ""
  var resort = ""
  IO.readLines(new FileInputStream("sunmar_cities.tsv")).foreach { line =>
    val spaces = line.takeWhile(_ == ' ').length
    val id = line.drop(spaces).takeWhile(_ != '\t')

    if (spaces == 2) {
      airport = id
      airports += id
    }
    if (spaces == 4) {
      resort = id
      resorts += id
      children.getOrElseUpdate(airport, mutable.ArrayBuffer.empty) += id
    }
    if (spaces == 6) {
      beaches += id
      children.getOrElseUpdate(resort, mutable.ArrayBuffer.empty) += id
    }
  }


  def updateCities(from: Seq[String], to: String, airportsFile: String) {
    val ltMapping = IO.readLines(new FileInputStream(to)).collect {
      case Tabbed("5", geoId, ltId) => geoId -> ltId
    }.toVector

    val ostrovokMapping = IO.readLines(new FileInputStream(to)).collect {
      case Tabbed("4", geoId, ostrovokId) => geoId -> ostrovokId
    }.toVector

    val mapping = from.flatMap(f => readGeoMapping(new File(f)))
      .groupBy(_.geoId).mapValues(_.reduce(_ + _))
      .values.toVector
      .sortBy(_.geoId.toString)

    IO.using(new PrintWriter(to)) { pw =>
      ltMapping.foreach { case (geoId, ltId) =>
        pw.println(Partners.lt.id + "\t" + geoId + "\t" + ltId)
      }
      mapping.foreach { m =>
        val sunmarIds = m.sunmarId.flatMap(_.split("-"))
        if (sunmarIds.nonEmpty) {
          val rs = sunmarIds.filter(resorts.contains)
          val bs = sunmarIds.filter(beaches.contains)

          val ids = if (rs.isEmpty) bs else rs.flatMap(children.getOrElse(_, Seq.empty))

          val cleanIds = ids.map(_.split("\\|").last).distinct
          if (cleanIds.nonEmpty)
            pw.println(Partners.sunmar.id + "\t" + m.geoId + "\t" + cleanIds.mkString(","))
        }
      }
      ostrovokMapping.foreach { case (geoId, ltId) =>
        pw.println(Partners.ostrovok.id + "\t" + geoId + "\t" + ltId)
      }
    }
    IO.using(new PrintWriter(airportsFile)) { pw =>
      mapping.foreach { m =>
        val sunmarIds = m.sunmarId.flatMap(_.split("-"))
        if (sunmarIds.nonEmpty) {
          val as = sunmarIds.filter(airports.contains)
          val cleanIds = as.map(_.split("\\|").last).distinct
          if (cleanIds.nonEmpty)
            pw.println(Partners.sunmar.id + "\t" + m.geoId + "\t" + cleanIds.mkString(","))
        }
      }
    }
  }
  
  def updateDepartures(from: String, to: String): Unit = {
    val ltMapping = IO.readLines(new FileInputStream(to)).collect {
      case Tabbed("5", geoId, ltId) => geoId -> ltId
    }.toSeq

    val sunmarMapping = IO.readLines(new FileInputStream(from)).collect {
      case Tabbed(IntValue(geoId), sunmarId, _) =>
        geoId -> sunmarId
    }.toSeq

    IO.using(new PrintWriter(to)) { pw =>
      ltMapping.foreach { case (geoId, ltId) =>
        pw.println(Partners.lt.id + "\t" + geoId + "\t" + ltId)
      }
      sunmarMapping.foreach { case (geoId, sunmarId) =>
        pw.println(Partners.sunmar.id + "\t" + geoId + "\t" + sunmarId)
      }
    }
  }

  def updateAirports(from: String, to: String): Unit = {
    val ltMapping = IO.readLines(new FileInputStream(to)).collect {
      case Tabbed("5", geoId, ltId) => geoId -> ltId
    }.toSeq

    val sunmarMapping = IO.readLines(new FileInputStream(from)).collect {
      case Tabbed(IntValue(geoId), sunmarId, _) =>
        geoId -> sunmarId
    }.toSeq

    IO.using(new PrintWriter(to)) { pw =>
      ltMapping.foreach { case (geoId, ltId) =>
        pw.println(Partners.lt.id + "\t" + geoId + "\t" + ltId)
      }
      sunmarMapping.foreach { case (geoId, sunmarId) =>
        pw.println(Partners.sunmar.id + "\t" + geoId + "\t" + sunmarId)
      }
    }
  }

  updateCities(
    Seq(
      "util-heavy/python/geo/data/matchingresults/geo2AllPartners.resorts.tsv",
      "util-heavy/python/geo/data/matchingresults/geo2AllPartners.resorts.dups.tsv"
    ),
    "tours-data/data/cities.tsv",
    "tours-data/data/airports.tsv"
  )

  /*updateDepartures(
    "util-heavy/python/geo/data/matchingresults/geo2Sunmar.departure.cities.tsv",
    "tours-data/data/departures.tsv"
  )*/

  sys.exit()
}
