package ru.yandex.tours.tools.ammo

import java.io.PrintWriter

import org.joda.time.LocalDate
import ru.yandex.tours.client.RelativeSearchRequest
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.Collections._

import scala.util.Random

object ApiAmmoGenerator extends Tool {
  private val interval = 2.5 //
  private val rps = 20
  private val windowSize = (rps.toDouble * interval).toInt

  implicit class RandomIterable[T](it: Iterable[T]) {
    def randomElement: T = {
      val i = Random.nextInt(it.size)
      it.drop(i - 1).head
    }
    def sample(count: Int) = {
      Random.shuffle(it).take(count)
    }
    def sample() = {
      Random.shuffle(it).take(it.size)
    }
    def shuffle = {
      Random.shuffle(it)
    }
  }
  implicit class RandomInt[T](i: Int) {
    def withDeviation(percent: Int) = {
      val diff = i.toDouble * percent / 100 * Random.nextGaussian()
      i + diff.toInt
    }
  }
  implicit class RandomDouble[T](d: Double) {
    def withDeviation(percent: Int) = {
      val diff = d * percent / 100 * Random.nextGaussian()
      d + diff
    }
  }

  private def regionChain(regionId: Int) = {
    for {
      region <- tree.region(regionId).toSeq
      r <- region :: tree.pathToRoot(region)
      if geoMapping.isKnownDestination(r.id)
    } yield r.id
  }

  private val hotelGeoMap = {
    for {
      hotel <- hotelsIndex.hotels
      regionId <- regionChain(hotel.geoId)
    } yield regionId -> hotel
  }.toSeq.toMultiMap

  val sessions =
  (1 to 7).flatMap(i => parseFile(s"tours-warmer/src/main/resources/serial_$i.txt")).map(request => {
    val x = request.toHotelSearchRequest(LocalDate.now.plusDays(30))
    val params = Seq(
      "to" -> x.to.toString,
      "from" -> x.from.toString,
      "nights" -> x.nights.toString,
      "when" -> x.when.toString,
      "when_flex" -> x.flexWhen.toString,
      "nights_flex" -> x.flexNights.toString,
      "lang" -> "ru"
    ) ++ x.ages.map("ages" -> _.toString)
    val cgi = params.map(t => s"${t._1}=${t._2}").mkString("&")

    val userId = Random.nextInt(10000000) + 1000000
    val recommendUri = s"/api/1.x/recommend?user_id=$userId&geo_id=${x.from}&lang=ru"
    val searchUri = s"/api/1.x/search/tours?$cgi"
    val minPriceUri = s"/api/1.x/statistic/min_prices?$cgi"
    val hotels = hotelGeoMap.getOrElse(x.to, Seq.empty).sample(2.3d.withDeviation(200).round.toInt max 0)

    val requests = Vector.newBuilder[String]

    if (Random.nextDouble() < 0.25) {
      requests += recommendUri
    }
    requests += minPriceUri
    for (i <- 0 until 6) {
      requests += (searchUri + "&request_index=" + i)
    }
    requests += minPriceUri

    for (hotel <- hotels) {
      val hotelUri = s"/api/1.x/hotel/${hotel.id}/info?lang=ru"
      val nearHotelsUri = s"/api/1.x/hotel/${hotel.id}/near_hotels?latitude=${hotel.latitude}&longitude=${hotel.longitude}&lon_span=0.051498413085930395&lat_span=0.015254949744022639&lang=ru"

      requests += hotelUri
      requests += nearHotelsUri

      val searchHotelUri = s"/api/1.x/hotel/${hotel.id}/search?$cgi"
      for (i <- 0 until 5) {
        requests += (searchHotelUri + "&request_index=" + i)
      }
    }

    requests.result()
  }).map(_.toIterator)
  val sample = sessions.sample(sessions.size / 20)

  var i = 0

  val pw = new PrintWriter("ammo.txt")
  var (window, rest) = sample.splitAt(windowSize)
  window = window.filter(_.hasNext)

  while (window.nonEmpty) {
    val it = window.randomElement
    pw.println(it.next())
    i += 1
    if (i % 1000 == 0) {
      println(s"$i: window.size = ${window.size} rest.size = ${rest.size}")
    }

    window = window.filter(_.hasNext)
    while (window.size < windowSize && rest.nonEmpty) {
      val (n, r) = rest.splitAt(windowSize - window.size)
      rest = r
      window = (window ++ n) filter (_.hasNext)
    }
  }

  pw.close()

  private def parseFile(file: String) = {
    scala.io.Source.fromFile(file).getLines().map(RelativeSearchRequest.parse).flatMap(_.toOption)
  }
}
