package ru.yandex.tours.tools.db

import java.io.File

import ru.yandex.tours.app.{Application, DefaultEnvironment}
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.model.DbPartnerHotel
import ru.yandex.tours.db.tables.Hotels
import ru.yandex.tours.db.{DefaultMySqlSupport, Transactions}
import ru.yandex.tours.hotels.HotelIO
import ru.yandex.tours.model.hotels.HotelsHolder
import ru.yandex.tours.model.hotels.HotelsHolder.{PartnerHotel, RawPartnerHotel}
import slick.driver.MySQLDriver.api._

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.duration._
import scala.concurrent.{Await, Future}
import scala.util.{Failure, Success}

object HotelToPartnerHotelConverter extends Application with DefaultEnvironment with DefaultMySqlSupport {

  val dao = new HotelsDao(db)

  def convert(hotel: HotelsHolder.Hotel): DbPartnerHotel = {
    val rawHotel = RawPartnerHotel.newBuilder()
      .addAllName(hotel.getNameList)
      .addAllAddress(hotel.getAddressList)
      .setStars(hotel.getStars)
      .addAllFeatures(hotel.getFeaturesList)
      .setPartner(hotel.getPartnerId)
      .setPartnerUrl(hotel.getPartnerUrl)
      .setPartnerId(hotel.getLocalId)
      .addAllPhone(hotel.getPhoneList)
      .addAllSynonyms(hotel.getSynonymsList)

    if (hotel.hasPoint) rawHotel.setPoint(hotel.getPoint)
    if (hotel.hasHotelUrl) rawHotel.setHotelUrl(hotel.getHotelUrl)
    if (hotel.hasRating) rawHotel.setRating(hotel.getRating)


    val builder = PartnerHotel.newBuilder()
      .setId(hotel.getId)
      .setRawHotel(rawHotel)
      .addAllImages(hotel.getImagesList)
      .setIsDeleted(false)

    if (hotel.hasType) builder.setType(hotel.getType)

    DbPartnerHotel(builder.getId, builder.build, isNew = false,
      builder.getIsDeleted, System.currentTimeMillis(), System.currentTimeMillis())
  }

  Await.result(db.createIfNotExists(Transactions.table), 1.minute)

  val used = mutable.HashSet.empty[Int]

  def deduplicate(hotels: Iterable[DbPartnerHotel]): Iterable[DbPartnerHotel] = {
    val result = mutable.Buffer.empty[DbPartnerHotel]
    for (hotel <- hotels) {
      if (!used.contains(hotel.id)) {
        result += hotel
        used += hotel.id
      }
    }
    result
  }

  def load(hotels: Iterator[DbPartnerHotel]): Future[Unit] = {
    val slice = deduplicate(hotels.take(500).toVector)
    if (slice.nonEmpty) {
      val q = Hotels.table ++= slice
      db.run(q).flatMap(_ => load(hotels))
    } else {
      Future.successful {}
    }
  }

  val hotelsToLoad = HotelIO.loadFromFileAsIterator(new File("sodis_final.proto")).flatMap(h => h :: h.getChildrenList.toList).filter(_.getPartnerId != 6).map(convert)
  load(hotelsToLoad) onComplete {
    case Success(_) => println("Success!")
    case Failure(e) => e.printStackTrace()
  }
}
