package ru.yandex.tours.tools.db

import org.apache.commons.codec.binary.Hex
import ru.yandex.tours.model.hotels.HotelsHolder
import ru.yandex.tours.model.hotels.HotelsHolder.{Hotel, PartnerHotel, RawPartnerHotel}
import ru.yandex.tours.tools.HotelAware
import ru.yandex.tours.util.{GZip, IO}

import scala.collection.JavaConversions._
import scala.collection.mutable

object HotelsToSqlScript extends App with HotelAware {
  def convertToPartner(hotel: HotelsHolder.Hotel) = {
    val rawHotel = RawPartnerHotel.newBuilder()
      .addAllName(hotel.getNameList)
      .addAllAddress(hotel.getAddressList)
      .setStars(hotel.getStars)
      .addAllFeatures(hotel.getFeaturesList)
      .setPartner(hotel.getPartnerId)
      .setPartnerUrl(hotel.getPartnerUrl)
      .setPartnerId(hotel.getLocalId)
      .addAllPhone(hotel.getPhoneList)
      .addAllSynonyms(hotel.getSynonymsList)

    if (hotel.hasPoint) rawHotel.setPoint(hotel.getPoint)
    if (hotel.hasHotelUrl) rawHotel.setHotelUrl(hotel.getHotelUrl)
    if (hotel.hasRating) rawHotel.setRating(hotel.getRating)


    val builder = PartnerHotel.newBuilder()
      .setId(hotel.getId)
      .setRawHotel(rawHotel)
      .addAllImages(hotel.getImagesList)
      .setIsDeleted(false)

    if (hotel.hasType) builder.setType(hotel.getType)
    builder.build
  }

  def convert(hotel: Hotel): String = {
    val bytes = convertToPartner(hotel).toByteArray
    val result = Seq(
      hotel.getId,
      "x'" + Hex.encodeHexString(GZip.compress(bytes)) + "'",
      false,
      false
    ).mkString(", ")
    "(" + result + ")"
  }

  val used = mutable.HashSet.empty[Int]

  def deduplicate(hotels: Iterable[Hotel]) = {
    val result = mutable.Buffer.empty[Hotel]
    for (hotel <- hotels) {
      if (!used.contains(hotel.getId)) {
        result += hotel
        used += hotel.getId
      }
    }
    result
  }

  IO.printFile("hotels.sql") { pw =>
    hotels("sodis_final.proto").flatMap(h => h :: h.getChildrenList.toList).filter(_.getPartnerId != 6).sliding(100, 100).foreach { batch =>
      val line = s"INSERT INTO hotels (id, payload, is_new, is_deleted) VALUES " + deduplicate(batch).map(convert).mkString(",") + ";"
      pw.println(line)
    }
  }
}
