package ru.yandex.tours.tools.geo

import org.apache.commons.lang3.StringUtils
import ru.yandex.tours.db.CustomRegions
import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.custom.{PartialRegion, CustomRegion}
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.model.{LocalizedString, Languages, MapRectangle}
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{IntValue, DoubleValue, Tabbed}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.text.StringNormalizer
import slick.driver.MySQLDriver.api._

import scala.collection.mutable
import scala.concurrent.Await
import scala.concurrent.duration.Duration
import scala.io.Source

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.12.15
 */
object RegionsImport extends Tool {

  val toAdd = Seq.newBuilder[CustomRegion]
  val toUpdate = Seq.newBuilder[(Int, MapRectangle)]

  val map = {
    data.regionTree.regions.flatMap { r =>
      r.boundingBox.coordinateIndexes.map(_ -> r)
    }.toMultiMap
  }

  val badBoxes = Set(11020, 11146, 10712, 11309, 40, 52, 10002, 11457)

  def area(bb: MapRectangle) = bb.latSpan * bb.lonSpan
  def r(r: Region) = s"[${r.id}:${r.name.ruName}]"

  Source.fromFile("/Users/darl/Downloads/ski_resorts_result.tsv").getLines().drop(1).foreach {
    case Tabbed(rawGeoId, name, DoubleValue(lat), DoubleValue(lon), DoubleValue(latSpan), DoubleValue(lonSpan), comment) =>
      val geoId = IntValue(rawGeoId)
      val isNew = geoId.isEmpty

      val rect = MapRectangle.byCenterAndSpan(lon, lat, lonSpan, latSpan)


      if (isNew) {
        val parents = map.getOrElse(
          (lon.toInt, lat.toInt),
          Seq.empty
        ).filter(_.boundingBox.contains(lon, lat))

        val bestParent =
          if (parents.nonEmpty) parents.minBy(r => area(r.boundingBox))
          else data.regionTree.root.get

        val nameNormalized = StringNormalizer.normalizeString(name)
        def equal(r: Region) = {
          val parentNormalized = StringNormalizer.normalizeString(r.name.ruName)
          (parentNormalized == nameNormalized ||
            parentNormalized.startsWith(nameNormalized) ||
            StringUtils.getLevenshteinDistance(nameNormalized, parentNormalized) <= 2) &&
            !(r.id == 105183 && name == "Восс")
        }

        val equalParent = parents.find { equal }

        val siblings = tree.allChildren(bestParent)
        val equalSibling = siblings.find(equal)


        val tz = tree.pathToRoot(bestParent).flatMap(_.timeZone).headOption


        if (equalParent.nonEmpty) {
          println(s"Region $name already exist: ${r(equalParent.get)}")
          toUpdate += (equalParent.get.id -> rect)
        } else if (equalSibling.nonEmpty) {
          println(s"Region $name already exist in siblings: ${r(equalSibling.get)}")
          toUpdate += (equalSibling.get.id -> rect)
        } else {
          println(s"For region $name best parent is ${tree.pathToRoot(bestParent).map(r).mkString(" <- ")}")

          toAdd += CustomRegion(
            id = 0,
            isNew = true,
            partialRegion = PartialRegion(
              `type` = Some(Types.Other),
              parentId = Some(bestParent.id),
              name = Some(LocalizedString(Languages.ru -> name)),
              genitive = Some(name),
              dative = Some(name),
              accusative = Some(name),
              locative = Some(name),
              preposition = Some("в"),
              longitude = Some(lon),
              latitude = Some(lat),
              boundingBox = Some(rect),
              timeZone = tz
            ),
            children = Seq.empty
          )
        }
      } else {
        for (region <- data.regionTree.region(geoId.get)) {
          val outside = data.regionTree.pathToRoot(region)
            .filter(_.boundingBox.nonEmpty)
            .filter(r => r != region)
            .filter(r => !badBoxes.contains(r.id))
            .find(!_.boundingBox.contains(lon, lat))
          if (outside.isDefined) {
            println(s"Region ${r(region)} is outside of ${r(outside.get)}. Region moved = ${geo.distanceInKm(region.latitude, region.longitude, rect.latCenter, rect.lonCenter)} km.")
          } else {
            toUpdate += (region.id -> rect)
          }
        }
      }
  }

  {
    val updated = toUpdate.result().toMap
    val visited = new mutable.HashSet[Int]()
    val lines = Source.fromFile("tours-data/data/region_boundaries.tsv").getLines().toList
    IO.printFile("tours-data/data/region_boundaries.tsv") { pw =>
      lines.foreach {
        case line @ Tabbed(IntValue(geoId), DoubleValue(minLon), DoubleValue(minLat), DoubleValue(maxLon), DoubleValue(maxLat)) =>
          updated.get(geoId) match {
            case Some(bbox) =>
              pw.println(geoId + "\t" + bbox.toTsv)
              visited += geoId
            case None => pw.println(line)
          }
      }
      updated.filter(kv => !visited.contains(kv._1)).foreach {
        case (geoId, bbox) =>
          pw.println(geoId + "\t" + bbox.toTsv)
      }
    }
  }

  val newRegions = toAdd.result()
  newRegions.foreach(println)
//  println(newRegions.size + " new regions")

//  Await.result(testDb.run(CustomRegions.table ++= newRegions), Duration.Inf)
//  Await.result(prodDb.run(CustomRegions.table ++= newRegions), Duration.Inf)
}
