package ru.yandex.tours.tools.hotels

import models.Hotel
import org.slf4j.LoggerFactory
import ru.yandex.tours.model.filter.hotel.{PartnerIdFilter, GeoIdFilter}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners._
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.Collections

import scala.io.Source

/**
  * Created by Evgeny Zhoga on 30.12.15.
  */
object CheckHotelsInUnknownRegionByPartner extends Tool {
  private val log = LoggerFactory.getLogger(getClass)
  var maxRegions = 0
  var maxHotels = 0
  def processPartner(partner: Partner, regionIds: List[Int]) = {

    def getSuperRegion(regionId: Int): Option[Int] =
      if (geoMapping.getPartnerDestination(partner, regionId).isEmpty)
        tree.parent(regionId).flatMap(p => getSuperRegion( p.id ))
      else Some(regionId)

    def getRegions(regionId: Int): List[Option[Int]] = {
      (if (geoMapping.getPartnerDestination(partner, regionId).isEmpty) Option.empty[Int]
      else Some(regionId)) :: tree.children(regionId).map(_.id).toList.flatMap(getRegions)
    }
    def getFirstRegions(regionId: Int): List[Option[Int]] = {
      if (geoMapping.getPartnerDestination(partner, regionId).isEmpty) Option.empty[Int] :: tree.children(regionId).map(_.id).toList.flatMap(getFirstRegions)
      else Some(regionId) :: Nil
    }
    def byGeoId3(geoId: Int, partner: Partner) =
      shardedHotelsIndex.getHotels(Paging(pageSize = 1000), new GeoIdFilter(geoId)).filter(h => h.partnerIds.toList.exists(_.partner == partner))

    def byGeoId2(geoId: Int, partner: Partner): Seq[Hotel] =
      shardedHotelsIndex.getHotels(Paging(pageSize = 1000), new GeoIdFilter(geoId)).filter(h => h.partnerIds.toList.exists(_.partner == partner)).toList :::
        tree.children(geoId).map(_.id).toList.flatMap(byGeoId2(_, partner))

    def byGeoId(geoId: Int, partner: Partner) = shardedHotelsIndex.getHotels(Paging(pageSize = 50), GeoIdFilter(geoId), PartnerIdFilter(partner.id))

    var foundOnPath = 0
    var foundOnPathWithSameRegion = 0
    var foundByHotels = 0
    var foundByExtendedHotels = 0
    var foundOnPathAndByHotels = 0
    var count = 0
    val unknownRegions = regionIds.flatMap { regionId =>
      val regionPath = getFirstRegions(regionId).flatten
      val superRegion = getSuperRegion(regionId)
      maxRegions = Math.max(maxRegions, regionPath.size)
      val hotels = byGeoId(regionId, partner).toList
      maxHotels = Math.max(maxHotels, hotels.size)

      val extendedHotels = byGeoId2(regionId, partner)
//      log.info(s"Partner $partner, region $regionId, found on path -> ${regionPath.mkString("[",", ", "]")}, found hotels -> ${hotels.map(_.id).mkString("[",", ", "]")} ")
      count += 1
      if (regionPath.nonEmpty) {
        log.info(s"Partner $partner, region $regionId, found on path -> ${regionPath.take(3).mkString("[",", ", " ... ]")}")
      }
      if (hotels.nonEmpty) {
        log.info(s"Partner $partner, region $regionId, found hotels -> ${hotels.take(3).map(_.id).mkString("[", ", ", " ... ]")}")
      }

      if (regionPath.nonEmpty && hotels.nonEmpty) {
        foundOnPathAndByHotels += 1
        None
      }
      else if (regionPath.nonEmpty) {
        foundOnPath += 1
        None
      }
      else if (hotels.nonEmpty) {
        foundByHotels += 1
        None
      } else Some(regionId)
    }
    log.info(s"Partner $partner, count: $count, foundOnPathAndByHotels: $foundOnPathAndByHotels, foundOnPath: $foundOnPath, foundByHotels: $foundByHotels")
    unknownRegions.toSet
  }

  val t = Source.fromFile("tours-data/partnerWithUnknownRegion").getLines().map{line =>
    val p = line.split("\\s+")
    p(0) -> p(1).toInt
  }.toList.groupBy(_._1).map(t => Partners.withName(t._1) -> t._2.map(_._2))

  val p2ur = t.map {case (partner, regionIds) =>
      partner -> processPartner(partner, regionIds)
  }

  p2ur.foreach { case (partner, unknownRegions) =>
      val un = p2ur.values.foldLeft(unknownRegions)((acc, next) => acc.intersect(next))
      log.info(s"$partner has ${unknownRegions.size} unknown regions including ${un.size} which are unknown to anybody")
  }

  log.info(s"Max subregion size is $maxRegions, max hotels size is $maxHotels")

}
