package ru.yandex.tours.tools.hotels

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.{CompositeDataDef, DataTypes}
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.hotels.enrichers.{GeoIdByPartnerHotelSetter, RegionWithConfidence}
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import shapeless.{::, HNil}

class ComplicatedCountryGeoIdBySpanSetter(tree: Tree) extends GeoIdByPartnerHotelSetter {
  private case class MapRectangleWithCountry(rectangle: MapRectangle, country: Region)

  private val countries = tree.regions.filter(_.isCountry)

  private val geoIndex2rectangles = tree.regions.flatMap { r =>
    for {
      country <- tree.country(r).toIterable
      if r.boundingBox.nonEmpty
      (i, j) <- r.boundingBox.coordinateIndexes
    } yield (i, j) -> MapRectangleWithCountry(r.boundingBox, country)
  }.groupBy(_._1).map {
    case (key, seq) => key -> seq.map(_._2)
  }.withDefaultValue(Iterable.empty[MapRectangleWithCountry])

  override def getGeoId(hotel: PartnerHotel): Option[RegionWithConfidence] = {
    if (hotel.getRawHotel.hasPoint) {
      val point = hotel.getRawHotel.getPoint
      getGeoIdByPoint(point.getLongitude, point.getLatitude)
    } else {
      None
    }
  }

  def getGeoIdByPoint(lon: Double, lat: Double): Option[RegionWithConfidence] = {
    val lonIndex = lon.toInt
    val latIndex = lat.toInt
    val countryCandidates = countries.filter(_.boundingBox.contains(lon, lat)).toSet
    val candidates: Iterable[MapRectangleWithCountry] = geoIndex2rectangles.apply((lonIndex, latIndex)).filter { mr =>
      countryCandidates.contains(mr.country)
    }
    val country2matchedCount = candidates.filter(_.rectangle.contains(lon, lat)).groupBy(_.country).map {
      case (countryId, seq) => countryId -> seq.size
    }
    if (country2matchedCount.isEmpty) {
      None
    } else {
      val max = country2matchedCount.values.max
      val best = country2matchedCount.filter(_._2 == max)
      if (best.size > 1) {
        None
      } else {
        Some(RegionWithConfidence(best.head._1, 0.5))
      }
    }
  }
}

object ComplicatedCountryGeoIdBySpanSetter extends CompositeDataDef[ComplicatedCountryGeoIdBySpanSetter, Tree :: HNil] {
  override def dependsOn: Set[DataType] = Set(DataTypes.regions)

  override def from(dependencies: Tree :: HNil): ComplicatedCountryGeoIdBySpanSetter = {
    val tree = dependencies.head
    new ComplicatedCountryGeoIdBySpanSetter(tree)
  }
}
