package ru.yandex.tours.tools.hotels

import java.io.{PrintWriter, File}

import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.hotels.{HotelsIndex, HotelIO}
import ru.yandex.tours.model.geo.MapObject
import ru.yandex.tours.model.hotels.HotelsHolder.TravelHotel
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.parsing.Tabbed
import ru.yandex.tours.model.Languages._

import scala.util.Try

object GeoIdClustering extends Tool {

  private val K = 3
  private val DIST_IN_KM = 5

  case class HotelInfo(id: Int, latitude: Double, longitude: Double, geoId: Int) extends MapObject {
    override def toString = {
      val region = tree.region(geoId).get
      val country = tree.country(region).get

      Tabbed(id, s"http://travel.yandex.ru/hotel/$id", geoId, region.name(ru), country.name(ru), latitude, longitude)
    }
  }

  val rawHotels: TraversableOnce[TravelHotel] = ???

  val hotels = rawHotels.filter(HotelsIndex.isIndexable).map { hotel =>
    HotelInfo(hotel.getId, hotel.getPoint.getLatitude, hotel.getPoint.getLongitude, hotel.getGeoId)
  }.toSet

  val (known, unknown) = hotels.partition { h =>
    val region = tree.region(h.geoId).get
    region.`type` == Types.SecondLevelCityDistrict ||
      region.`type` == Types.Village ||
      region.`type` == Types.City ||
      region.`type` == Types.CityDistrict
  }

  val map = known.groupBy(h => (h.latitude.toInt, h.longitude.toInt))

  val pw = new PrintWriter("fixed_geo_id.tsv")
  val failedPw = new PrintWriter("failed_geo_id.tsv")
  val conflicts = new PrintWriter("conflicts_geo_id.tsv")
  unknown.foreach { slave =>
    val candidates = for {
      i <- -1 to 1
      j <- -1 to 1
    } yield {
      val lonIndex = slave.longitude.toInt + i match {
        case x if x >= -180 && x <= 180 => x
        case -181 => 179
        case 181 => -179
      }
      val latIndex = slave.latitude.toInt + j
      map.getOrElse((latIndex, lonIndex), Seq.empty)
    }.filter(c => geo.distanceInKm(slave, c) <= DIST_IN_KM)
    val nearest = candidates.flatten.sortBy(c => geo.distanceInKm(slave, c)).take(K)
    if (nearest.isEmpty) {
      failedPw.println(slave)
    } else {
      val topGeoId = nearest.head.geoId
      if (nearest.forall(_.geoId == topGeoId)) {
        val region = tree.region(topGeoId).get
        val country = tree.country(region).get
        pw.println(Tabbed(slave, topGeoId, region.name(ru), country.name(ru), (0 until K).map(i => Try(nearest(i)).map(h => s"http://travel.yandex.ru/hotel/${h.id}").getOrElse("")).mkString("\t")))
      } else {
        conflicts.println(slave)
      }
    }
  }
  pw.close()
  failedPw.close()
  conflicts.close()

  // DECOMMENT THIS TO GATHER STATISTIC
  //  val notCity = hotels.count { h =>
  //    val region = tree.region(h.geoId).get
  //    !(region.`type` == Types.City || region.`type` == Types.Village)
  //  }
  //  val bag = new Bag[Type]()
  //  var interestingRegions = Set.empty[Region]
  //  hotels.foreach { h =>
  //    val region: Region = tree.region(h.geoId).get
  //    val tupe: Type = region.`type`
  //    if (tupe == Types.SecondLevelCityDistrict ||tupe == Types.CityDistrict ||tupe == Types.Airport ||tupe == Types.MetroStation) {
  //      interestingRegions += region
  //    }
  //    bag += tupe
  //  }
  //  println("Interesting regions:")
  //  interestingRegions.foreach { region =>
  //    println(Tabbed(region.name(Languages.ru), region.`type`, region.id))
  //  }
  //  println("By type: ")
  //  bag.entriesDesc.foreach {
  //    case (tupe, count) => println(tupe, count)
  //  }
  //  println("------")
  //  println(notCity)
  //  println(hotels.size)
}
