package ru.yandex.tours.tools.hotels

import java.io.{File, PrintWriter}
import java.util.concurrent.atomic.AtomicInteger

import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.provider.{HttpExtDataClient, RemoteExtDataService}
import ru.yandex.extdata.provider.cache.LocalFSDataCache
import ru.yandex.tours.extdata.{DataTypes, ExtDataUpdateChecker}
import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.hotels.{HotelRatings, ShardedYoctoHotelsIndex}
import ru.yandex.tours.model.{Languages, MapRectangle}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.util.file._
import ru.yandex.tours.util.parsing.Tabbed

import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 17.12.15
 */

trait GeoInputData {
  lazy val root = new File("indexer-data")
  root.mkdirs()
  private lazy val extDataClient = new HttpExtDataClient("http://localhost:36445", 4, 10000)
  private lazy val localFSCache = {
    val cacheDir = root / "cache"
    cacheDir.mkdirs()
    val cache = new LocalFSDataCache(
      DataTypes.registry,
      extDataClient,
      cacheDir.getPath,
      false
    )
    cache.afterPropertiesSet()
    cache
  }

  lazy val localFSDataService: ExtDataService = new RemoteExtDataService(localFSCache)
  lazy val regionTree = Tree.from(localFSDataService)
  lazy val hotelRatings = HotelRatings.from(localFSDataService)

  lazy val shardedHotelsIndex = ShardedYoctoHotelsIndex.fromExtData(
    new File("tours-data/hotels_index"),
    hotelRatings,
    localFSDataService,
    ExtDataUpdateChecker.empty,
    lazyLoadResources = false
  )
}

object GeoIdFixByNeighbours extends App with GeoInputData {

  val i = new AtomicInteger(0)
  val pw = new PrintWriter(root / "hotel_geo_id_by_neighbour.tsv")
  val cw = new PrintWriter(root / "hotel_geo_id_by_neighbour_candidates.tsv")

  MapRectangle(-179, -90, 180, 90).coordinateIndexes.foreach {
    processCoords
  }
  pw.close
  cw.close

  private def processCoords(coords: (Int, Int)) = {
    val (lon, lat) = coords
    val hotels = shardedHotelsIndex.inRectangle(MapRectangle.byCenterAndSpan(lon, lat, 3d, 3d), Int.MaxValue)
    val core = hotels.filter(h => h.latitude.toInt == lat && h.longitude.toInt == lon)
    if (core.nonEmpty) {
      println(s"Processing ($lon, $lat) with ${core.size} hotels")
    }
    core.par.foreach {
      processHotel(_, hotels)
    }
  }

  private def processHotel(hotel: Hotel, hotels: Seq[Hotel]) = {
    if (i.incrementAndGet % 1000 == 0) println(s"Processed ${i.get} hotels")
    try {
      val candidates = getCandidates(hotel, hotels)
      if (candidates.nonEmpty) {
        val best: Region = getBest(hotel, candidates)
        if (hotel.geoId != best.id) {
          if (candidates.size > 1) {
            cw.println(Tabbed(hotel.id, hotel.geoId, candidates.map(_.id).mkString(", ")))
          }
          saveAmending(hotel, best)
        }
      }
    } catch {
      case NonFatal(t) =>
    }
  }

  private def saveAmending(hotel: Hotel, best: Region) = {
    val region = regionTree.region(hotel.geoId).get
    val bestRegionPath = regionPath(best)
    val currentRegionPath = regionPath(region)
    val fits = region.boundingBox.contains(best.boundingBox)

    pw.println(Tabbed(
      hotel.id,
      best.id, bestRegionPath, best.`type`,
      s"https://www.google.ru/maps/place/${hotel.latitude}+${hotel.longitude}",
      hotel.geoId, currentRegionPath, region.`type`
    ))
  }

  private def getBest(hotel: Hotel, candidates: Set[Region]): Region = {
    val best = candidates.minBy { region => region.boundingBox.area }
    best
  }

  private def getCandidates(hotel: Hotel, hotels: Seq[Hotel], distance: Double = 5): Set[Region] = {
    val region = regionTree.region(hotel.geoId).get
    val country = regionTree.country(region).get
    val path = regionTree.pathToRoot(region).toSet

    val neighbours = hotels
      .filter(h => geo.distanceInKm(h, hotel) < distance)
      .sortBy(geo.distanceInKm(_, hotel))

    val near = neighbours.filter(h => geo.distanceInKm(h, hotel) < 0.2d).map(_.geoId).toSet
    val around = neighbours.groupBy(_.geoId).filter {
      case (geoId, hs) =>
        hs.exists(_.longitude > hotel.longitude) &&
        hs.exists(_.longitude < hotel.longitude) &&
        hs.exists(_.latitude > hotel.latitude) &&
        hs.exists(_.latitude < hotel.latitude)
    }.keySet

    val regions = (near ++ around).flatMap(regionTree.region)

    val candidates = regions.filter { r =>
      (r.boundingBox.nonEmpty && r.boundingBox.contains(hotel) && r.boundingBox.area < region.boundingBox.area) &&
        (regionTree.country(r).get == country) &&
        (path.subsetOf(regionTree.pathToRoot(r).toSet) || region.boundingBox.contains(r.boundingBox))
    }
    candidates
  }

  private def regionPath(region: Region) = {
    regionTree.pathToRoot(region).map(_.name(Languages.ru)).mkString(", ")
  }
}
