package ru.yandex.tours.tools.hotels.hotelindexing

import java.io.File
import java.util.zip.GZIPOutputStream

import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.provider.{HttpExtDataClient, RemoteExtDataService}
import ru.yandex.extdata.provider.cache.LocalFSDataCache
import ru.yandex.tours.app.{AkkaSupport, Application, DefaultEnvironment}
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.geo.partners.{PartnerRegionHotelCounts, PartnerRegionsDef}
import ru.yandex.tours.hotels.{HotelRatings, ShardedYoctoHotelsIndex, YoctoHotelsIndex}
import ru.yandex.tours.hotels.enrichers.{CountryGeoIdBySpanSetter, GeoIdByPartnerRegionSetter, MetaGeoIdSetter}
import ru.yandex.tours.indexer.hotels.HotelsIndexer
import ru.yandex.tours.model.hotels.HotelsHolder.TravelHotel
import ru.yandex.tours.tools.TestDb
import ru.yandex.tours.util.{IO, ProtoIO}
import ru.yandex.tours.util.file._
import shapeless.HNil

import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.{Failure, Success}


trait InputData {
  lazy val root = new File("indexer-data")
  root.mkdirs()
  private lazy val extDataClient = new HttpExtDataClient("http://localhost:36445", 4, 10000)
  private lazy val localFSCache = {
    val cacheDir = root / "cache"
    cacheDir.mkdirs()
    val cache = new LocalFSDataCache(
      DataTypes.registry,
      extDataClient,
      cacheDir.getPath,
      false
    )
    cache.afterPropertiesSet()
    cache
  }

  lazy val extDataService: ExtDataService = new RemoteExtDataService(localFSCache)
  lazy val geoMapping = GeoMappingHolder.from(extDataService)
  lazy val hotelRatings = HotelRatings.from(extDataService)
  lazy val regionTree = Tree.from(extDataService)
}

object HotelIndexOnly
  extends Application
    with DefaultEnvironment
    with TestDb
    with InputData
    with AkkaSupport {

  implicit val as = akkaSystem
  implicit val ec = as.dispatcher

  val hotelsDao = new HotelsDao(testDb)

  val partnerTree = PartnerRegionsDef.from(PartnerRegionHotelCounts.empty :: HNil, extDataService)
  val backaPermalinks = BackaPermalinks.from(extDataService)
  val partnerRegionSetter = new GeoIdByPartnerRegionSetter(geoMapping, partnerTree, regionTree)
  val countrySetter = CountryGeoIdBySpanSetter.from(regionTree :: HNil)
  val meta = new MetaGeoIdSetter(partnerRegionSetter, countrySetter)

  val indexer = new HotelsIndexer(testDb, hotelsDao, meta, regionTree, hotelRatings, backaPermalinks, null, 12.hours)


  def updateIndex: Future[_] = indexer.retrieveTravelHotels().map { file =>
    try {
      IO.deleteFile(root / "index")
    } finally {
      (root / "index").mkdirs()
    }
    try {
      for (shard <- 0 until ShardedYoctoHotelsIndex.SHARDS_COUNT) {
        // TODO get rid of this strange 24x reading of the same file...
        val toIndex = ProtoIO.loadFromFile(file, TravelHotel.PARSER).filter(indexer.getShardId(_, regionTree) == shard)
        val index = IO.usingTmp(s"hotel_shard_$shard") { os =>
          IO.using(new GZIPOutputStream(os)) { gzipOs =>
            YoctoHotelsIndex.build(toIndex, regionTree, hotelRatings, gzipOs)
          }
        }
        try {
          val dt = DataTypes.shardedHotels(shard)
          index.renameTo(root / "index" / (dt.getName + "-4-0"))
        } finally {}
      }
    } finally IO.deleteFile(file)
  } andThen {
    case Success(_) => log.info("HotIdx: Successfully indexed hotels!")
    case Failure(e) => log.warn("HotIdx: Can not index hotels!", e)
  }

  val result = updateIndex
  result.onComplete {
    case Success(_) =>
      println("Done!")
      sys.exit()
    case Failure(e) => e.printStackTrace()
  }
}
