package ru.yandex.tours.tools.images

import java.io.{File, PrintWriter}
import java.util.concurrent.ArrayBlockingQueue
import java.util.concurrent.atomic.LongAdder

import ru.yandex.tours.avatars.HttpAvatarClient
import ru.yandex.tours.model.hotels.HotelsHolder.Hotel
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.http.NingHttpClient
import ru.yandex.tours.util.parsing.Tabbed

import scala.collection.JavaConverters._
import scala.concurrent.Await
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.duration._
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 01.09.15
 */
object ImagesLoader extends Tool {
  assert(args.length == 3, "3 arguments should be specified: 1. Path to image urls. 2. partner name. 3. Number of worker threads")

  val threadNumber = args(2).toInt
  val httpClient = new NingHttpClient(None)
  val avatarClient = new HttpAvatarClient(httpClient, "http://localhost:8801", "http://avatars.mds.yandex.net", "tours")

  val queue = new ArrayBlockingQueue[String](1000)
  val downloaded = new LongAdder()
  val failed = new LongAdder()

  val successLog = new PrintWriter(new File(s"success.${args(1)}.tsv"))
  val failedLog = new PrintWriter(new File(s"failure.${args(1)}.tsv"))

  def getAllImages: Iterator[String] = {
    scala.io.Source.fromFile(args(0)).getLines()
  }
  
  val total = getAllImages.size
  println("Total images: " + total)

  def getImages(hotel: Hotel): Seq[String] = {
    val childImages = hotel.getChildrenList.asScala
      .flatMap(getImages)
    (childImages ++ hotel.getPhotosList.asScala).filter(_.startsWith("https://cdn.ostrovok.ru"))
  }

  def download(url: String): Unit = {
    try {
      val image = Await.result(
        avatarClient.put(url, ImageProviders.lt),
        30.seconds
      )
      downloaded.increment()
      successLog.println(Tabbed(
        url,
        image.host,
        image.group,
        image.name
      ))
      successLog.flush()
    } catch {
      case NonFatal(e) =>
        println(s"Failed to download $url: ${e.getMessage}")
        failed.increment()
        failedLog.println(url)
        failedLog.flush()
    }
  }


  new Thread("pusher") {
    override def run(): Unit = {
      getAllImages.foreach(queue.put)
    }
  }.start()

  def starWorker(i: Int): Unit = {
    new Thread("worker-" + i) {
      override def run(): Unit = {
        while(true) {
          val i = queue.take()
          download(i)
        }
      }
    }.start()
  }

  (0 until threadNumber).foreach(starWorker)

  val started = System.currentTimeMillis()
  new Thread("stats-logger") {
    override def run(): Unit = {
      while (true) {
        Thread.sleep(5000)
        val now = System.currentTimeMillis()
        val elapsed = (now - started).millis.toMinutes
        println(s"Downloaded ${downloaded.intValue()} images out of $total in $elapsed minutes. Failed: ${failed.intValue}")
      }
    }
  }.start()
}
