package ru.yandex.tours.tools.merging

import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import ru.yandex.tours.util.{IO, Logging}

trait CorpusAware extends Logging {

  sealed trait LearningCorpus {
    def comment: String

    def name: String
  }

  case object Hotels101Corpus extends LearningCorpus {
    override def comment: String =
      """
        |От ручного матчинга контентов 101 отеля. Чтобы его достать, смотри Hotels101MatchingExtractor
      """.stripMargin

    override def name: String = "101hotels_corpus.tsv"
  }

  case object DefaultLearningCorpus extends LearningCorpus {
    override def comment: String =
      """
        |Взяли спорные данные из толоки, где трое сказали, что отели разные, а двое, что одинаковые.
        |Смотри HOTELS-1542 и results2.xlsx.
      """.stripMargin

    override def name: String = "learning_corpus.tsv"
  }

  case object ExpediaInnaCorpus extends LearningCorpus {
    override def comment: String =
      """
        |Рита руками разметила. Смотри https://st.yandex-team.ru/HOTELS-1656#1461068854000
      """.stripMargin

    override def name: String = "expedia_inna_corpus.tsv"
  }

  case object ManualFromAdminCorpus extends LearningCorpus {
    override def comment: String =
      """
        |Все ручные матчинги из админки. Смотри ManualAdminMatchingExtractor
      """.stripMargin

    override def name: String = "manual_admin_corpus.tsv"
  }

  case object NearHotelCorpus extends LearningCorpus {
    override def comment: String =
      """
        |Рассматриваем отели, которые находятся рядом и не скластеризованы. Считаем их разными.
        |Смотри NearHotelExtractor
      """.stripMargin

    override def name: String = "near_hotel_corpus.tsv"
  }
  case object NearHotelCorpus2 extends LearningCorpus {
    override def comment: String =
      """
        |Еще один корпус на основе сущестующего индекса.
        |Для каждого отеля берем один положительный пример и до 5 отрицательных.
        |
        |Смотри NearHotelExtractor
      """.stripMargin

    override def name: String = "near_hotel_corpus_2.tsv"
  }

  case class Link(master: Int, slave: Int, isMerge: Boolean, corpus: LearningCorpus)

  case class LinkWithHotel(from: PartnerHotel, to: PartnerHotel, isMerge: Boolean, corpus: LearningCorpus)

  val allCorpuses = Seq(
    Hotels101Corpus,
    DefaultLearningCorpus,
    ExpediaInnaCorpus,
    ManualFromAdminCorpus,
    NearHotelCorpus
//    ,NearHotelCorpus2
  )

  def readCorpuses(corpuses: LearningCorpus*): Iterable[Link] = {
    for {
      corpus <- corpuses
      Tabbed(IntValue(child), IntValue(parent), IntValue(isMerge)) <- IO.readLines(s"data/${corpus.name}")
    } yield {
      val a = child min parent
      val b = child max parent
      Link(a, b, isMerge > 0, corpus)
    }
  }


  def getHotelLinks(contains: Int ⇒ Boolean, get: Int ⇒ Option[PartnerHotel],
                    corpuses: LearningCorpus*): Iterable[LinkWithHotel] = {
    val rawLinks = readCorpuses(corpuses: _*)
    val unknownHotels = rawLinks.flatMap(l => Set(l.slave, l.master)).filterNot(contains)
    if (unknownHotels.nonEmpty) {
      log.warn(s"Can not get hotels ${unknownHotels.size}: ${unknownHotels.mkString(", ")}")
    }
    rawLinks.flatMap(l => (get(l.master), get(l.slave), l.corpus) match {
      case (Some(from), Some(to), corpus) => Some(LinkWithHotel(from, to, l.isMerge, corpus))
      case _ => None
    })
  }
}
