package ru.yandex.tours.tools.osm

import java.io.{File, FileOutputStream, PrintWriter}
import java.util.concurrent.ThreadPoolExecutor.CallerRunsPolicy
import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.{ArrayBlockingQueue, ThreadPoolExecutor, TimeUnit}

import akka.util.Timeout
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.hotels.HotelIO
import ru.yandex.tours.model.Languages
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.http.NingHttpClient
import ru.yandex.tours.util.lang._
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}
import spray.http.Uri

import scala.concurrent.Await
import scala.concurrent.duration._
import scala.util.Try
import scala.util.control.NonFatal
import scala.concurrent.ExecutionContext.Implicits.global

object OsmDownloader extends Tool {
  assert(args.length == 2, "1. Hotels proto file 2. Threads number")

  val lang = Languages.ru
  private val API_KEY = "jhXQi9e02Ax3iLhUQDmJvWw2GLQkYM6Y"
  private val successFile = s"success.osm_$lang.tsv"
  private val failureFile = s"failure.osm_$lang.tsv"

  val downloadedRegions: Set[Int] = Try(scala.io.Source.fromFile(successFile).getLines().map {
    case line@Tabbed(IntValue(geoId), response) =>
      geoId
  }.toSet).getOrElse {
    println(s"Failed to read previous success: $successFile. Start from beginning")
    Set.empty
  }

  val interestingGeoIds = {
    val hotelGeoIds = HotelIO.loadFromFileAsIterator(new File(args(0))).map(_.getGeoId)
    hotelGeoIds.toSet ++
      geoMapping.countryGeoIds ++
      geoMapping.cityGeoIds ++
      geoMapping.departuresGeoIds
  }

  val interestingRegions = interestingGeoIds.flatMap { geoId =>
    tree.pathToRoot(geoId)
  }

  val countryCodes = scala.io.Source.fromFile("util-heavy/data/hotels/booking/country.code.name.tsv").getLines().map {
    case Tabbed(name, cc) => name.toLowerCase -> cc
  }.toMap

  val country2code = tree.regions.filter(_.isCountry).flatMap { region =>
    countryCodes.get(region.name(Languages.en).toLowerCase).map(region -> _) onEmpty {
      println(s"Can not find country code for: ${region.name(Languages.ru)}, ${region.name(Languages.en)}, ${region.id}")
    }
  }.toMap

  println("Total interesting regions: " + interestingRegions.size)


  val success = new PrintWriter(new FileOutputStream(new File(successFile), true))
  val fail = new PrintWriter(failureFile)
  val downloaded = new AtomicInteger()
  val failed = new AtomicInteger()

  def download(region: Region) = {
    val url = Uri("http://open.mapquestapi.com/nominatim/v1/search.php").withQuery(
      "key" -> API_KEY,
      "format" -> "json",
      "q" -> region.name(lang),
      "polygon" -> "1",
      "countrycodes" -> country2code.getOrElse(region, "")
    )
    try {
      val (sc, result) = Await.result(httpClient.get(url)(Timeout(30.seconds)), 30.seconds)
      assert(sc.intValue == 200, s"Non 200 response: $sc: $result")
      val doneCount = downloaded.incrementAndGet()
      if (doneCount % 1000 == 0) {
        println(s"Downloaded: $doneCount. Failed: ${failed.get()}")
      }
      success.println(Tabbed(region.id, result.replaceAll("\\s", " ")))
      success.flush()
    } catch {
      case NonFatal(e) =>
        fail.println(Tabbed(region.name, e.getMessage))
        fail.flush()
        failed.incrementAndGet()
    }
  }

  val threadsNumber = args(1).toInt
  val httpClient = new NingHttpClient(None)
  val queue = new ArrayBlockingQueue[Runnable](1000)
  val pool = new ThreadPoolExecutor(threadsNumber, threadsNumber, 10, TimeUnit.MINUTES, queue, new CallerRunsPolicy)
  val regionsToDownload = interestingRegions.filterNot(r => downloadedRegions.contains(r.id))
  println(s"We are going to download ${regionsToDownload.size} regions")
  regionsToDownload.foreach { region =>
    pool.submit(new Runnable {
      override def run(): Unit = download(region)
    })
  }
}
