package ru.yandex.tours.tools

import play.api.libs.json._
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.MapRectangle
import ru.yandex.tours.tools.osm.MemberTypes._
import ru.yandex.tours.util.parsing.IntValue
import ru.yandex.tours.util.parsing.Tabbed
import ru.yandex.tours.util.lang._
import ru.yandex.tours.model.Languages._

import scala.util.Try

package object osm {
  case class OsmObject(id: Long,
                       lat: Double,
                       lon: Double,
                       address: Map[String, String],
                       importance: Double,
                       clazz: String,
                       `type`: String,
                       osmType: MemberType,
                       hasPolygon: Boolean,
                       rectangle: MapRectangle) {
    def url = {
      s"http://www.openstreetmap.org/${osmType.toString.toLowerCase}/$id"
    }
  }

  object OsmObject {
    def fromJson(json: String): Try[OsmObject] = {
      fromJson(Json.parse(json).as[JsObject])
    }

    def fromJson(json: JsObject): Try[OsmObject] = Try {
      val `type` = (json \ "type").as[String]
      if (`type` == "postcode") {
        throw new Exception("Post code is not interesting for us")
      }
      val id = json \ "osm_id" match {
        case JsNumber(x) => x.toLong
        case JsString(x) => x.toLong
        case t => throw new Exception(s"Can not parse osm id: $t")
      }
      val lat = parseDouble(json \ "lat")
      val lon = parseDouble(json \ "lon")
      val address = (json \ "address").asOpt[JsObject].map(_.value.map { case (key, value) => key -> value.as[String] }.toMap).getOrElse(Map.empty)
      val importance = parseDouble(json \ "importance")
      val clazz = (json \ "class").as[String]
      val osmType = MemberTypes.withName((json \ "osm_type").as[String].toUpperCase)
      val hasPolygon = json.keys.contains("polygonpoints")
      val bounding = (json \ "boundingbox").as[JsArray].value.map(parseDouble)
      val rectangle = MapRectangle.byBoundaries(bounding(2), bounding.head, bounding(3), bounding(1))
      OsmObject(id, lat, lon, address, importance, clazz, `type`, osmType, hasPolygon, rectangle)
    }
  }

  private def parseDouble(js: JsValue) = js match {
    case JsNumber(x) => x.toDouble
    case JsString(x) => x.toDouble
    case _ => throw new Exception(s"Can not cast $js to double")
  }

  def parseOsm(lang: Lang, fits: OsmObject => Boolean) = {
    scala.io.Source.fromFile(s"success.osm_$lang.tsv").getLines().flatMap {
      case Tabbed(IntValue(geoId), json) =>
        val res = for {
          tryParsed <- Try(Json.parse(json)).toOption.toIterable
          jsObject <- tryParsed.as[JsArray].value
          osm <- OsmObject.fromJson(jsObject.as[JsObject]).onFailure(e => println(s"Can not parse some osm $geoId: ${e.getMessage}")).toOption
          if fits(osm)
        } yield osm
        Some(geoId -> res)
      case line =>
        val parts = line.split("\t")
        println("Can not parse: " + parts(0))
        None
    }.filter(_._2.nonEmpty).toMap
  }

  def similarity(osmObject: OsmObject, geoId: Int, tree: Tree) = {
    val geoNames = tree.pathToRoot(geoId).flatMap(_.allNames)
    osmObject.address.values.count(geoNames.contains)
  }
}
