package ru.yandex.tours.tools.sunmar

import java.io.PrintWriter

import ru.yandex.tours.partners.sunmar.{ToPlace, OTICredential, SearchService, DefaultSunmarClient}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.Tabbed

import scala.collection.mutable
import scala.concurrent.Await
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 29.05.15
 */
object SunmarGeoTool extends App {
  val searchService = new SearchService().getBasicHttpEndpoint
  val cred = new OTICredential

  cred.setUsername("Yandex")
  cred.setPassword("65536")

  val client = new DefaultSunmarClient(searchService, cred)

  val areas = Await.result(client.fromAreas, 30.seconds)
  val countries = Await.result(client.countries, 30.seconds)
  val places = Await.result(client.allToPlaces, 300.seconds).distinctBy(_.getID) //.map(fixPlaceID)

  val countryMap = countries.map(c => c.getID -> c).toMap

  def fixPlaceID(place: ToPlace) = {
    new ToPlace()
      .withID(place.getID.split("\\|").last)
      .withParentID(place.getParentID.split("\\|").last)
      .withCountry(place.getCountry)
      .withFromArea(place.getFromArea)
      .withName(place.getName)
  }

  IO.using(new PrintWriter("sunmar_departures.tsv")) { out =>
    for (area <- areas) {
      out.println(Tabbed(
        area.getID,
        area.getLname,
        area.getRegion.getLname,
        area.getRegion.getCountry.getLname
      ))
    }
  }

  IO.using(new PrintWriter("sunmar_countries.tsv")) { out =>
    for (country <- countries) {
      out.println(Tabbed(
        country.getID,
        country.getLname
      ))
    }
  }

  IO.using(new PrintWriter("sunmar_cities.tsv")) { out =>
    val placesMap = places.map(p => p.getID -> p).toMap
    val parent = new mutable.HashMap[String, String]()
    for (place <- places) {
      if (place.getParentID != "0") {
        assert(!parent.contains(place.getID))
        parent(place.getID) = place.getParentID
      }
    }
    val children = parent.toSeq.map(_.swap).toMultiMap

    def printRecursively(place: ToPlace, ident: Int): Unit = {
      if (ident >= 4) println(s"Ident = $ident, place = ${place.getID}" )
      out.println(("  " * ident) + Tabbed(
        place.getID,
        place.getName
      ))
      for (kidId <- children.getOrElse(place.getID, Seq.empty).sortBy(k => placesMap(k).getName)) {
        val kid = placesMap(kidId)
        if (kid.getFromArea == place.getFromArea && kid.getCountry == place.getCountry) {
          printRecursively(kid, ident + 1)
        }
      }
    }

    for (place <- places if !parent.contains(place.getID)) {
      val country = countryMap(place.getCountry)
      out.println(Tabbed(country.getID, country.getLname))

      printRecursively(place, 1)
    }
  }

  sys.exit()
}
