package ru.yandex.tours.tools.wizard

import org.joda.time.LocalDate
import ru.yandex.tours.model.Languages
import ru.yandex.tours.tools.Tool
import ru.yandex.tours.util.parsing.IntValue
import ru.yandex.tours.wizard.domain.{TourStartInterval, ToursWizardRequest}

import scala.collection.mutable
import scala.sys.process._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.04.15
 */
object NotFoundLogParser extends Tool {
  val TWR = ".*ToursWizardRequest\\((.+),([0-9]+),(Some\\([0-9]+\\)|None),Set\\((.*)\\),TourStartInterval\\((.+),(.+)\\),([0-9]+),(Some\\([0-9]+\\)|None),(Some\\([0-9]+\\)|None),(Some\\([0-9]+\\)|None)\\)".r

  val ru = Languages.ru

  val lines =
    "ssh logger01h.vs.yandex.net \"/home/darl/wizard_logs.sh\"".lineStream.iterator

  object OptValue {
    def unapply(str: String): Option[Option[String]] = {
      str match {
        case "None" => Some(None)
        case _ if str.startsWith("Some(") && str.endsWith(")") => Some(Some(str.stripPrefix("Some(").stripSuffix(")")))
        case _ => None
      }
    }
  }

  val requests = lines.map {
    case TWR(userRequest, IntValue(userRegion), OptValue(regionInRequest), markers,
      intervalStart, intervalEnd, IntValue(from), OptValue(optTo), OptValue(optHotelId), OptValue(optOperatorId)) =>
      val to = optTo.map(_.toInt)
      val hotelId = optHotelId.map(_.toInt)
      val operatorId = optOperatorId.map(_.toInt)
      val start = LocalDate.parse(intervalStart)
      val end = LocalDate.parse(intervalEnd)

      ToursWizardRequest(userRequest, 0, None, Set.empty, TourStartInterval(start, end), from, Seq.empty, to, hotelId, operatorId)
  }.toVector

  private def printSamples(reqs: Seq[ToursWizardRequest]) = {
    val occ = new mutable.HashMap[String, Int] { override def default(k: String) = 0 }
    for (y <- reqs) occ(y.originalRequest.stripSuffix(" ")) += 1
    occ.toSeq.sortBy(-_._2).take(5).mkString(", ")
  }

  def anyTour(): Unit = {
    println("[Хочу в отпуск!] – колдунщик направлений")
    requests.view
      .filter(_.to.isEmpty)
      .filter(_.hotel.isEmpty)
      .filter(_.operator.isEmpty)
      .groupBy(_.from)
      .toVector
      .sortBy(-_._2.size)
      .foreach { case (from, reqs) =>
      val region = tree.region(from).get
      println(s" * ${region.name(ru)} - ${reqs.size} (${printSamples(reqs)})")
    }
  }

  def anyTourWithOperator(): Unit = {
    println("[Хочу в отпуск только с ним] – колдунщик направлений c заданным оператором")
    requests.view
      .filter(_.to.isEmpty)
      .filter(_.hotel.isEmpty)
      .filter(_.operator.nonEmpty)
      .groupBy(r => (r.from, r.operator.get))
      .toVector
      .sortBy(-_._2.size)
      .foreach { case ((from, operatorId), reqs) =>
      val region = tree.region(from).get
      val operatorName = operators.getById(operatorId).fold("Unknown")(_.name)
      println(s" * $operatorName ${region.name(ru)} - ${reqs.size} (${printSamples(reqs)})")
    }
  }

  def toursToCountry(): Unit = {
    println("[Хочу сюда] – колдунщик отелей")
    requests.view
      .filter(_.to.nonEmpty)
      .filter(_.hotel.isEmpty)
      .filter(_.operator.isEmpty)
      .groupBy(r => (r.from, r.to.get))
      .toVector
      .sortBy(-_._2.size)
      .foreach { case ((from, to), reqs) =>
      val region = tree.region(from).get
      val toRegion = tree.region(to).get
      println(s" * ${toRegion.name(ru)} из ${region.genitive} - ${reqs.size} (${printSamples(reqs)})")
    }
  }
  def toursToCountryWithOperator(): Unit = {
    println("[Хочу туда] – колдунщик отелей с заданным оператором")
    requests.view
      .filter(_.to.nonEmpty)
      .filter(_.hotel.isEmpty)
      .filter(_.operator.nonEmpty)
      .groupBy(r => (r.from, r.to.get, r.operator.get))
      .toVector
      .sortBy(-_._2.size)
      .foreach { case ((from, to, operatorId), reqs) =>
      val region = tree.region(from).get
      val toRegion = tree.region(to).get
      val operatorName = operators.getById(operatorId).fold("Unknown")(_.name)
      println(s" * ${toRegion.name(ru)} из ${region.genitive} от $operatorName - ${reqs.size} (${printSamples(reqs)})")
    }
  }
  def toursInHotel(): Unit = {
    println("[Хочу туда] – колдунщик отеля")
    requests.view
      .filter(_.hotel.nonEmpty)
      .groupBy(r => r.hotel.get)
      .toVector
      .sortBy(-_._2.size)
      .foreach { case (hotelId, reqs) =>
      val hotelName = hotelsIndex.getHotelById(hotelId).fold("Unknown")(_.name(ru))
      println(s" * $hotelName - ${reqs.size} (${printSamples(reqs)})")
    }
  }
  def toursWithHotelMarker(): Unit = {
    println("[Хочу] – запросы с отельным маркером")
    requests.view
      .filter(_.hasHotelMarker)
      .filter(_.hotel.isEmpty)
      .filter(_.to.isEmpty)
      .groupBy(r => r.from)
      .toVector
      .sortBy(-_._2.size)
      .foreach { case (from, reqs) =>
      val region = tree.region(from).get
      println(s" * из ${region.genitive} - ${reqs.size} (${printSamples(reqs)})")
    }
  }

//  anyTour()
  anyTourWithOperator()
//  toursToCountry()
//  toursToCountryWithOperator()
//  toursInHotel()
//  toursWithHotelMarker()
}
