package ru.yandex.tours

import akka.actor.Actor
import com.codahale.metrics.MetricRegistry
import com.google.common.util.concurrent.RateLimiter
import ru.yandex.tours.client.{RelativeSearchRequest, ToursClient}
import ru.yandex.tours.supplier.RequestSupplier
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.zoo.SharedValue

/**
 * Created with IntelliJ IDEA.
 * User: Anton Ivanov <antonio@yandex-team.ru>
 * Date: 06.02.15
 * Time: 15:53
 */

sealed trait RequestDispatcherMessage
case class SuccessSendFresh(request: RelativeSearchRequest) extends RequestDispatcherMessage
case class SuccessSendFromCache(request: RelativeSearchRequest) extends RequestDispatcherMessage
case class ErrorSend(ex: Throwable) extends RequestDispatcherMessage

class RequestDispatcherActor(metrics: MetricRegistry, supplier: RequestSupplier, client: ToursClient, isStable: Boolean, rpsValue: SharedValue[Double])
  extends Actor with Logging {

  val successMeter = metrics.meter("warmer.dispatcher.success")
  val fromCacheMeter = metrics.meter("warmer.dispatcher.cache")
  val errorMeter = metrics.meter("warmer.dispatcher.error")

  override def preStart(): Unit = {
    sendWithRateLimiter()
  }

  override def receive: Receive = {
    case SuccessSendFresh(uri) =>
      log.info(s"success load $uri without cache. Wait for next sending for some time..")
      successMeter.mark()
      sendWithRateLimiter()
    case SuccessSendFromCache(uri) =>
      fromCacheMeter.mark()
      if (isStable) {
        log.info(s"success load $uri from cache -> send next request without wait")
        sendNextRequest()
      } else {
        log.info(s"success load $uri from cache -> send next request with wait (testing or dev)")
        sendWithRateLimiter()
      }
    case ErrorSend(ex) =>
      errorMeter.mark()
      log.error(s"error in tours searcher", ex)
      sendWithRateLimiter() // use rater to not flood if searcher is down
  }

  private def sendWithRateLimiter(): Unit = {
    rateLimiter.acquire()
    sendNextRequest()
  }

  private[this] def sendNextRequest(): Unit = {
    val nextUri = supplier.nextRequest()
    client.checkRequest(nextUri, self)
  }

  private[this] val rateLimiter = RateLimiter.create(rpsValue.get)
  log.info(s"Started with RPS: " + rpsValue.get)
  rpsValue.onUpdate { newRps =>
    log.info(s"Updated RPS: $newRps")
    rateLimiter.setRate(newRps)
  }
}
