package ru.yandex.tours.client

import org.joda.time.{LocalDate, MonthDay}
import ru.yandex.tours.util.parsing.IntValue

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 07.03.15
 */
trait WarmerDate {
  def toLocalDate(base: LocalDate = LocalDate.now()): LocalDate
  def serialize: String
}

object WarmerDate {
  val Fixed = "DATE_(\\d+)\\.(\\d+)".r
  val Shift = "SHIFT_(\\d+)".r
  val Static = "STATIC_(.+)".r

  def parse(str: String): WarmerDate = {
    str match {
      case Fixed(IntValue(day), IntValue(month)) => FixedDate(new MonthDay(month, day))
      case Shift(IntValue(shift)) => ShiftDate(shift)
      case IntValue(shift) => ShiftDate(shift)
      case Static(date) => StaticDate(LocalDate.parse(date))
    }
  }
}

case class FixedDate(md: MonthDay) extends WarmerDate {
  override def toLocalDate(base: LocalDate): LocalDate = {
    val localDate = md.toLocalDate(base.getYear)
    if (localDate.isBefore(base)) localDate.plusYears(1)
    else localDate
  }
  def serialize = s"DATE_${md.getDayOfMonth}.${md.getMonthOfYear}"
}

case class ShiftDate(shiftDays: Int) extends WarmerDate {
  override def toLocalDate(base: LocalDate): LocalDate = base.plusDays(shiftDays)
  override def serialize: String = "SHIFT_" + shiftDays
}

case class StaticDate(date: LocalDate) extends WarmerDate {
  override def toLocalDate(base: LocalDate): LocalDate = date
  override def serialize: String = "STATIC_" + date
}