package ru.yandex.tours.supplier

import java.util.Date

import org.joda.time.DateTime
import ru.yandex.tours.client.RelativeSearchRequest
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang._

import scala.collection.immutable.TreeMap
import scala.util.{Random, Success}

/**
 * Created with IntelliJ IDEA.
 * User: Anton Ivanov <antonio@yandex-team.ru>
 * Date: 10.02.15
 * Time: 10:55
 */
case class RandomWithPrioritySupplier(elements: TreeMap[Int, RelativeSearchRequest], total: Int, seed: Long = new Date().getTime) extends RequestSupplier with Logging {
  override def nextRequest(current: DateTime): RelativeSearchRequest = {
    val randomPriority = random.nextInt(total)
    val (priority, rawElement) = elements.to(randomPriority).last
    log.debug(s"RandomWithPrioritySupplier: for randomPriority=$randomPriority select element with priority $priority")
    rawElement
  }

  val random = new Random(seed)
}

object RandomWithPrioritySupplierBuilder extends FileSupplierBuilder[RandomWithPrioritySupplier] {
  override def parseFileForDayOfWeek(dayOfWeek: Int): RandomWithPrioritySupplier = {
    val fileName = s"priority_$dayOfWeek.txt"
    var prioritySum = 0

    val priorityToRequest = allLines(fileName).map(_.split('@')).filter(_.size == 2).map(parts =>
      RelativeSearchRequest.parse(parts(0)).map(request => {
        val priority = parts(1).toInt
        require(priority >= 0 && priority <= 100, s"wrong priority $priority Wait values in [0, 100]")
        priority -> request
      }) onFailure {
        case e => log.error(s"skip wrong line [${parts.mkString("@")}] from $fileName: ${e.getMessage}")
      }
    ).collect({case Success((priority, request)) =>
      val prevSum = prioritySum
      prioritySum += priority
      prevSum -> request
    })
    val map = TreeMap[Int, RelativeSearchRequest](priorityToRequest.toSeq: _*)
    log.info(s"success load file $fileName with total elements ${map.size} and total sum=$prioritySum")
    RandomWithPrioritySupplier(map, prioritySum)
  }
}