package ru.yandex.tours.supplier

import akka.actor.{ActorSystem, Props}
import com.google.common.collect.{MinMaxPriorityQueue, Queues}
import org.joda.time.DateTime
import ru.yandex.tours.client.RelativeSearchRequest
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.prices.PriceSearcher
import ru.yandex.tours.storage.subscriptions.SubscriptionStorage
import ru.yandex.tours.util.{Logging, Metrics}
import ru.yandex.tours.warmer.SubscriptionsAmmoActor

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.11.15
 */
class SubscriptionsSupplier(subscriptionStorage: SubscriptionStorage,
                            priceSearcher: PriceSearcher,
                            hotelsIndex: HotelsIndex,
                            fallbackSupplier: RequestSupplier)
                           (implicit akkaSystem: ActorSystem) extends RequestSupplier with Logging {

  private val metrics = Metrics("warmer.subscriptions.supplier")

  private val queue = MinMaxPriorityQueue
    .maximumSize(5000)
    .expectedSize(5000)
    .create[SubscriptionsAmmoActor.Bullet]()

  private val synchronized = Queues.synchronizedQueue(queue)
  private val props = Props(new SubscriptionsAmmoActor(subscriptionStorage, priceSearcher, hotelsIndex, synchronized, 20.minutes))
  akkaSystem.actorOf(props, "subscription-ammo-generator")

  metrics.newGauge("age") {
    val bullet = synchronized.peek()
    if (bullet eq null) 0L
    else (DateTime.now().getMillis - bullet.lastSearch.getMillis).millis.toMinutes.toDouble / 60
  }

  metrics.newGauge("queue") {
    synchronized.size()
  }

  override def nextRequest(current: DateTime): RelativeSearchRequest = {
    val bullet = synchronized.poll()
    if (bullet eq null) fallbackSupplier.nextRequest(current)
    else {
      log.info(s"Using bullet $bullet")
      bullet.request
    }
  }
}
