package ru.yandex.tours.warmer

import java.util

import akka.actor.Actor
import akka.pattern.pipe
import org.joda.time.DateTime
import ru.yandex.tours.calendar.Calendar.FlightDay
import ru.yandex.tours.client.{RelativeSearchRequest, StaticDate}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.search.SearchDates
import ru.yandex.tours.model.subscriptions.Subscription
import ru.yandex.tours.model.util.{CustomDateInterval, CustomNights}
import ru.yandex.tours.prices.PriceSearcher
import ru.yandex.tours.storage.subscriptions.SubscriptionStorage
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.lang.Futures

import scala.collection.mutable
import scala.collection.JavaConverters._
import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.11.15
 */
class SubscriptionsAmmoActor(subscriptionStorage: SubscriptionStorage,
                             priceSearcher: PriceSearcher,
                             hotelsIndex: HotelsIndex,
                             queue: util.Queue[SubscriptionsAmmoActor.Bullet],
                             checkInterval: FiniteDuration = 20.minutes) extends Actor with Logging {

  import SubscriptionsAmmoActor._
  import context.dispatcher

  private val maxAge = 7.days

  case object Reload
  case class Subscriptions(subscriptions: Seq[Subscription])
  case class Bucket(from: Int, to: Int, ages: Seq[Int])

  def generateAmmo(bucket: Bucket, days: Seq[FlightDay]): Future[Unit] = {
    val Bucket(from, to, ages) = bucket
    val (minDate, maxDate) = days.map(_.when).minMax
    val nights = days.map(_.nights).distinct
    val dates = SearchDates(CustomDateInterval(minDate, maxDate), CustomNights(nights), None)
    priceSearcher.getFreshness(from, to, ages, dates, maxAge).map { freshness =>
      log.info(s"Loaded bucket (from = $from, to = $to, ages = $ages) with oldest price = ${freshness.values.minOpt}")
      val fallbackDate = DateTime.now().minusMillis(maxAge.toMillis.toInt)
      val bullets = for (day <- days) yield {
        val lastSearch = freshness.getOrElse(day, fallbackDate)
        val request = RelativeSearchRequest(from, to, day.nights, StaticDate(day.when), ages, flexWhen = false, flexNights = false)
        Bullet(request, lastSearch)
      }
      log.info(s"Generated ${bullets.size} bullets for bucket (from = $from, to = $to, ages = $ages) with oldest bullet = ${bullets.map(_.lastSearch).minOpt}")
      queue.addAll(bullets.asJava)
      ()
    }.andThen {
      case Failure(ex) =>
        log.warn(s"Failed to load bucket (from = $from, to = $to, ages = $ages)", ex)
    }
  }

  @throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    super.preStart()
    context.system.scheduler.scheduleOnce(0.seconds, self, Reload)
  }

  override def receive: Receive = {
    case Reload =>
      subscriptionStorage.getAllSubscriptions
        .map(Subscriptions.apply)
        .pipeTo(self)

    case Subscriptions(subscriptions) =>
      val data = new mutable.HashMap[Bucket, mutable.Set[FlightDay]]()
      for (s <- subscriptions if s.enabled) {
        val params = s.subject.params
        val to: Int = s.subject match {
          case Subscription.Direction(_, geoId, _) => geoId
          case Subscription.Hotel(_, hotelId, _) => hotelsIndex.getHotelById(hotelId).fold(0)(_.geoId)
        }

        val key = Bucket(params.from, to, params.ages)
        val set = data.getOrElseUpdate(key, mutable.HashSet.empty[FlightDay])
        for {
          date <- params.dates.when.values
          nights <- params.dates.nights.values
        } {
          set += FlightDay(date, nights)
        }
      }
      queue.clear()
      log.info(s"Reloading freshness for ${data.size} buckets")
      Futures.lazySequence {
        for ((bucket, days) <- data.iterator) yield {
          generateAmmo(bucket, days.toSeq)
        }
      }.andThen {
        case _ =>
          context.system.scheduler.scheduleOnce(checkInterval, self, Reload)
      }
  }
}

object SubscriptionsAmmoActor {

  case class Bullet(request: RelativeSearchRequest, lastSearch: DateTime) extends Comparable[Bullet] {
    override def compareTo(o: Bullet): Int = lastSearch.compareTo(o.lastSearch)
  }
}