package ru.yandex.tours.index

import java.io.{OutputStream, Closeable, File, FileOutputStream}
import java.nio.ByteBuffer

import org.apache.commons.io.output.ClosedOutputStream
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.util.lang.Dates._
import org.joda.time.LocalDate

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 06.03.15
 */
class IndexWriter(os: OutputStream) extends Closeable {
  def this(file: File) = this(new FileOutputStream(file))

  def writeHeader(size: Int): Unit = {
    writeHeader(WizardIndexing.formatVersion, System.currentTimeMillis(), size)
  }

  def writeHeader(freshness: Long, size: Int): Unit = {
    writeHeader(WizardIndexing.formatVersion, freshness, size)
  }

  def writeHeader(formatVersion: Int, freshness: Long, size: Int): Unit = {
    val header = ByteBuffer.allocate(WizardIndexing.headerSize)
    header.putInt(formatVersion)
    header.putLong(freshness)
    header.putInt(size)
    os.write(header.array())
  }

  def writeItem(operatorId: Int, from: Int, to: Int, when: Int, nights: Int, hotelId: Int, minPrice: Int, pansion: Pansion): Unit = {
    val buffer = ByteBuffer.allocate(WizardIndexing.recordSize)
    buffer.putInt(operatorId)
    buffer.putInt(from)
    buffer.putInt(to)
    buffer.putInt(when)
    buffer.putInt(nights)
    buffer.putInt(hotelId)
    buffer.putInt(minPrice)
    buffer.putInt(pansion.getNumber)
    os.write(buffer.array())
  }

  final def writeItem(operatorId: Int, from: Int, to: Int, when: LocalDate, nights: Int, hotelId: Int, minPrice: Int, pansion: Pansion): Unit = {
    writeItem(operatorId, from, to, when.toCompactInt, nights, hotelId, minPrice, pansion)
  }

  final def writeItem(item: WizardIndexItem): Unit = {
    writeItem(
      item.operatorId,
      item.from,
      item.to,
      item.when,
      item.nights,
      item.hotelId,
      item.minPrice,
      Pansion.valueOf(item.pansion)
    )
  }

  def close(): Unit = {
    os.close()
  }
}

class ShardedIndexWriter(writers: IndexedSeq[IndexWriter]) extends IndexWriter(ClosedOutputStream.CLOSED_OUTPUT_STREAM) {
  require(writers.size == WizardIndexing.shardCount, s"Expected ${WizardIndexing.shardCount} writers, got ${writers.size}")
  override def writeHeader(formatVersion: Int, freshness: Long, size: Int): Unit = {
    writers.foreach {
      _.writeHeader(formatVersion, freshness, -1)
    }
  }

  override def writeItem(operatorId: Int, from: Int, to: Int, when: Int, nights: Int, hotelId: Int, minPrice: Int, pansion: Pansion): Unit = {
    val i = WizardIndexing.shardId(operatorId, from, to)
    writers(i).writeItem(operatorId, from, to, when, nights, hotelId, minPrice, pansion)
  }

  override def close(): Unit = {
    writers.foreach(_.close())
  }
}