package ru.yandex.tours.index

import java.nio.ByteBuffer

import org.joda.time.LocalDate
import ru.yandex.tours.util.lang.Dates._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 06.03.15
 */
abstract class WizardIndexItem extends Comparable[WizardIndexItem] {
  def operatorId: Int
  def from: Int
  def to: Int
  def when: Int
  def nights: Int
  def hotelId: Int

  def minPrice: Int
  def pansion: Int

  override def compareTo(o: WizardIndexItem): Int = {
    val compareOperator = Ordering.Int.compare(this.operatorId, o.operatorId)
    if (compareOperator != 0) return compareOperator
    val compareFrom = Ordering.Int.compare(this.from, o.from)
    if (compareFrom != 0) return compareFrom
    val compareTo = Ordering.Int.compare(this.to, o.to)
    if (compareTo != 0) return compareTo
    val compareWhen = Ordering.Int.compare(this.when, o.when)
    if (compareWhen != 0) return compareWhen
    val compareNights = Ordering.Int.compare(this.nights, o.nights)
    if (compareNights != 0) return compareNights
    val compareHotelId = Ordering.Int.compare(this.hotelId, o.hotelId)
    if (compareHotelId != 0) return compareHotelId
    0
  }
}
object WizardIndexItem {
  case class SimpleIndexItem(operatorId: Int, from: Int, to: Int,
                             when: Int, nights: Int, hotelId: Int, minPrice: Int, pansion: Int) extends WizardIndexItem

  def apply(operatorId: Int, from: Int, to: Int, when: LocalDate, nights: Int, hotelId: Int, minPrice: Int, pansion: Int): WizardIndexItem =
    SimpleIndexItem(operatorId, from, to, when.toCompactInt, nights, hotelId, minPrice, pansion)

  def apply(operatorId: Int, from: Int, to: Int, when: Int, nights: Int, hotelId: Int, minPrice: Int, pansion: Int): WizardIndexItem =
    SimpleIndexItem(operatorId, from, to, when, nights, hotelId, minPrice, pansion)

  def apply(item: WizardIndexItem): WizardIndexItem = item match {
    case simple: SimpleIndexItem => simple
    case _ => SimpleIndexItem(item.operatorId, item.from, item.to, item.when, item.nights, item.hotelId, item.minPrice, item.pansion)
  }

  def fromBuffer(buffer: ByteBuffer): WizardIndexItem = {
    val operatorId = buffer.getInt
    val from = buffer.getInt
    val to = buffer.getInt
    val when = buffer.getInt
    val nights = buffer.getInt
    val hotelId = buffer.getInt
    val minPrice = buffer.getInt
    val pansion = buffer.getInt
    apply(operatorId, from, to, when, nights, hotelId, minPrice, pansion)
  }
}