package ru.yandex.tours.index.shard

import java.io.File
import java.nio.ByteBuffer

import org.joda.time.{DateTime, LocalDate}
import ru.yandex.tours.index.{WizardIndex, WizardIndexItem, WizardIndexIterator, WizardIndexing}
import ru.yandex.tours.util.io.ByteBuffers
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.wizard.resource.{ExpiringWizardResource, WizardResourceLoader}

import scala.annotation.tailrec

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.02.15
 *
 * @param formatVersion format version
 * @param freshness time when index shard was created
 * @param size count of elements in shard
 * @param buffer sorted array of (operatorId: Int, from: Int, to: Int, date: Int, nights: Int, hotelId: Int, minPrice: Int, pansion: Int) tuples
 */
class IndexShard(val formatVersion: Int, val freshness: Long, val size: Int, buffer: ByteBuffer)
  extends WizardIndex with ExpiringWizardResource {

  def this() = this(WizardIndexing.formatVersion, 0, 0, ByteBuffer.wrap(Array.emptyByteArray))

  require(formatVersion == WizardIndexing.formatVersion, "Unknown format version: " + formatVersion)
  require(buffer.limit() == size * WizardIndexing.recordSize, "Unexpected size of buffer: " + buffer.limit())

  private val ord = Ordering.apply[WizardIndexItem]

  private def findPosition(operatorId: Int, from: Int, to: Int, date: LocalDate): Int = {
    val buffer = this.buffer.duplicate()
    val tuple = WizardIndexItem(operatorId, from, to, date, 0, 0, 0, 0)

    @tailrec
    def binSearch(start: Int, end: Int): Int = {
      if (start >= end) return start
      val position = (end + start) / 2
      buffer.position(position * WizardIndexing.recordSize)

      val t = WizardIndexItem.fromBuffer(buffer)
      val cmp = ord.compare(t, tuple)

      if (cmp < 0) binSearch(position + 1, end)
      else binSearch(start, position)
    }

    binSearch(0, size)
  }

  override def find(operatorId: Int, from: Int, to: Int,
                    fromDate: LocalDate, untilDate: LocalDate): WizardIndexIterator = {
    val pos = findPosition(operatorId, from, to, fromDate)
    val until = untilDate.toCompactInt
    WizardIndexIterator.fromIterator {
      new IndexShardIterator(buffer.duplicate(), pos, size)
        .takeWhile(i => i.operatorId == operatorId && i.from == from && i.to == to && i.when < until)
    }
  }

  override def iterator: WizardIndexIterator = {
    new IndexShardIterator(buffer.duplicate(), 0, size)
  }

  override def toString: String = s"IndexShard($formatVersion, ${new DateTime(freshness)}, $size)"
}

object IndexShard extends WizardResourceLoader[IndexShard] {

  override def fromFile(file: File): IndexShard = {
    val buffer = ByteBuffers.mmap(file)
    val formatVersion = buffer.getInt
    val minTs = buffer.getLong
    val size = buffer.getInt

    val fixedSize = if (size >= 0) size else buffer.slice().limit / WizardIndexing.recordSize

    new IndexShard(formatVersion, minTs, fixedSize, buffer.slice()) {
      override def delete(): Unit = file.delete()
      override def load(): Unit = buffer.load()
      override def close(): Unit = ByteBuffers.unmap(buffer)
    }
  }

  override def empty: IndexShard = new IndexShard()

  def fromBuffer(buffer: ByteBuffer): IndexShard = {
    buffer.position(0)
    val formatVersion = buffer.getInt
    val minTs = buffer.getLong
    val size = buffer.getInt
    val fixedSize = if (size >= 0) size else buffer.slice().limit / WizardIndexing.recordSize

    new IndexShard(formatVersion, minTs, fixedSize, buffer.slice())
  }
}
