package ru.yandex.tours.query

import com.google.common.hash.BloomFilter
import it.unimi.dsi.fastutil.longs.Long2IntOpenHashMap
import org.joda.time.MonthDay

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.02.15
 */
sealed trait Pragmatic

case object Unknown extends Pragmatic
case object Ignored extends Pragmatic

sealed trait Marker extends Pragmatic
case object TourMarker extends Marker
case object HotMarker extends Marker
case object BuyMarker extends Marker
case object SearchMarker extends Marker
case class HotelMarker(hotelType: QueryHotelType) extends Marker
case object OfficialSiteMarker extends Marker
case object PriceMarker extends Marker
case object ReviewsMarker extends Marker
case object PhotosMarker extends Marker
case object NoVisa extends Marker
case object CheapMarker extends Marker
case class Stars(count: Int) extends Marker
case object SkiResorts extends Marker

case object StopWord extends Pragmatic

sealed trait GeoRegion extends Pragmatic {
  def regionId: Int
}

case class SomeRegion(regionId: Int) extends GeoRegion
case class DepartureRegion(regionId: Int) extends GeoRegion
case class ArrivalRegion(regionId: Int) extends GeoRegion

case class TourOperatorMarker(operatorId: Int) extends Pragmatic

case class HotelName(hotelId: Int) extends Pragmatic
case class HotelNameTop(hotelId: Int) extends Pragmatic

sealed trait HotelNamePart extends Pragmatic {
  def contains(nameId: Long): Boolean
}
case class HotelNamePartMap(hotels: Long2IntOpenHashMap) extends HotelNamePart {
  override def contains(nameId: Long): Boolean = hotels.containsKey(nameId)

  override def toString: String =
    if (hotels.size < 5) s"HotelNamePart($hotels)"
    else s"HotelNamePart(${hotels.size} hotels)"
}
case class HotelNamePartBloom(bloomFilter: BloomFilter[java.lang.Long]) extends HotelNamePart {
  override def contains(nameId: Long): Boolean = bloomFilter.mightContain(nameId)
}

case class DateInterval(from: MonthDay, until: MonthDay) extends Pragmatic

object Pragmatic {
  import java.nio.ByteBuffer

  import com.esotericsoftware.kryo.io.{ByteBufferInputStream, Input, Output}
  import com.esotericsoftware.kryo.pool.{KryoCallback, KryoFactory, KryoPool}
  import com.esotericsoftware.kryo.{Kryo, Serializer}
  import com.twitter.chill.KryoBase
  import ru.yandex.tours.util.IO

  private val kryoPool = new KryoPool.Builder(new KryoFactory {
    override def create(): Kryo = {
      val kryo = new KryoBase
      kryo.setInstantiatorStrategy(new org.objenesis.strategy.StdInstantiatorStrategy)
      kryo.setRegistrationRequired(true)
      kryo.setReferences(false)
      kryo.registerClasses(Seq(
        Unknown, Ignored, TourMarker, HotMarker, BuyMarker, SearchMarker, OfficialSiteMarker, PriceMarker,
        ReviewsMarker, PhotosMarker, NoVisa, CheapMarker, SkiResorts, StopWord
      ).map(_.getClass))
      kryo.register(classOf[HotelMarker])
      kryo.register(classOf[Stars])
      kryo.register(classOf[SomeRegion])
      kryo.register(classOf[DepartureRegion])
      kryo.register(classOf[ArrivalRegion])
      kryo.register(classOf[TourOperatorMarker])
      kryo.register(classOf[HotelName])
      kryo.register(classOf[HotelNamePartMap])
      kryo.register(classOf[DateInterval])
      kryo.register(classOf[Long2IntOpenHashMap], new Serializer[Long2IntOpenHashMap] {
        override def write(kryo: Kryo, output: Output, obj: Long2IntOpenHashMap): Unit = {
          output.writeInt(obj.size())
          val it = obj.long2IntEntrySet().fastIterator()
          while (it.hasNext) {
            val e = it.next()
            output.writeLong(e.getLongKey)
            output.writeInt(e.getIntValue)
          }
        }
        override def read(kryo: Kryo, input: Input, tpe: Class[Long2IntOpenHashMap]): Long2IntOpenHashMap = {
          val size = input.readInt()
          val map = new Long2IntOpenHashMap()
          var i = 0
          while (i < size) {
            map.put(input.readLong(), input.readInt())
            i += 1
          }
          map
        }
      })
      kryo.register(classOf[QueryHotelType])
      kryo.register(classOf[MonthDay], new Serializer[MonthDay]() {
        override def write(kryo: Kryo, output: Output, obj: MonthDay): Unit = {
          output.writeString(obj.toString)
        }
        override def read(kryo: Kryo, input: Input, `type`: Class[MonthDay]): MonthDay = {
          MonthDay.parse(input.readString())
        }
      })
      kryo.register(classOf[HotelNamePartBloom])
      kryo.register(classOf[Array[Long]])
      kryo.register(classOf[BloomFilter[_]])
      //scalastyle:off classforname
      kryo.register(Class.forName("com.google.common.hash.BloomFilterStrategies$BitArray"))
      kryo.register(Class.forName("com.google.common.hash.BloomFilterStrategies"))
      kryo.register(Class.forName("com.google.common.hash.Funnels$LongFunnel"))
      kryo.register(classOf[HotelNameTop])
      kryo
    }
  }).build()

  def read(bb: ByteBuffer): Pragmatic = {
    kryoPool.run(new KryoCallback[Pragmatic] {
      override def execute(kryo: Kryo): Pragmatic =
        kryo.readClassAndObject(new Input(new ByteBufferInputStream(bb))).asInstanceOf[Pragmatic]
    })
  }

  def write(pragmatic: Pragmatic): Array[Byte] = {
    kryoPool.run(new KryoCallback[Array[Byte]] {
      override def execute(kryo: Kryo): Array[Byte] =
        IO.using(new Output(1024, 1024 * 1024 * 16)) { output =>
          kryo.writeClassAndObject(output, pragmatic)
          output.toBytes
        }
    })
  }
}