package ru.yandex.tours.query.parser

import org.joda.time.MonthDay
import org.joda.time.format.DateTimeFormat
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.model.util.Util
import ru.yandex.tours.model.util.Util.HotelNamePart
import ru.yandex.tours.query.{HotelName, HotelNameTop, Pragmatic}
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.naming.HotelNameId
import ru.yandex.tours.util.parsing.{IntValue, Tabbed}

import scala.collection.JavaConverters._
import scala.io.Source

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.02.15
 */
trait ParserResources {
  def getStopWords: Seq[String]

  def getIgnoreWords: Seq[String]

  def getMarkers: Map[String, List[String]]

  def getHotelParts: Iterator[(String, Map[HotelNameId, Int])]

  def getOperators: Map[Int, List[String]]

  def getRegions: Map[Int, List[String]]

  def getDateIntervals: Map[(MonthDay, MonthDay), List[String]]

  def getReqAns: Iterator[(String, Pragmatic)]

  def getAliases: collection.Map[Int, Seq[String]]
}

class LocalParserResources(extDataService: ExtDataService) extends ParserResources {

  private val formatter = DateTimeFormat.forPattern("dd.MM")

  private def getStream(dataType: DataType) = extDataService.readData(dataType)

  private def loadLines(dataType: DataType): Iterator[String] = {
    val stream = getStream(dataType)
    Source.fromInputStream(stream)
      .getLines()
      .map(_.trim)
      .filterNot(_.startsWith("#"))
      .filter(_.nonEmpty)
      .bindTo(stream)
  }

  private def loadMapping(dataType: DataType): Map[String, List[String]] = loadLines(dataType).collect {
    case Tabbed(key, value) => key -> value
  }.toIterable.toMultiMap

  private def loadIMapping(dataType: DataType): Map[Int, List[String]] = loadLines(dataType).collect {
    case Tabbed(IntValue(id), name) => id -> name
  }.toIterable.toMultiMap

  override def getStopWords: Seq[String] = loadLines(DataTypes.stopWords).toSeq

  override def getIgnoreWords: Seq[String] = loadLines(DataTypes.ignoreWords).toSeq

  override def getMarkers: Map[String, List[String]] = loadMapping(DataTypes.wizardMarkers)

  override def getHotelParts: Iterator[(String, Map[HotelNameId, Int])] = {
    def convertNameMap(namePart: HotelNamePart): Map[HotelNameId, Int] = {
      namePart.getHotelList.asScala
        .filter(_.getWords > 1)
        .map(h => HotelNameId(h.getHotelId, h.getNameId) -> h.getWords)
        .toMap
    }
    val stream = getStream(DataTypes.hotelNameParts)
    Iterator.continually(Util.HotelNamePart.parseDelimitedFrom(stream))
      .takeWhile(_ ne null)
      .map { namePart => (namePart.getPhrase, convertNameMap(namePart)) }
      .filter(_._2.nonEmpty)
      .bindTo(stream)
  }

  override def getOperators: Map[Int, List[String]] = {
    loadIMapping(DataTypes.operatorAliases)
  }

  override def getRegions: Map[Int, List[String]] = {
    loadIMapping(DataTypes.wizardGeo)
  }

  override def getDateIntervals: Map[(MonthDay, MonthDay), List[String]] = {
    loadLines(DataTypes.dates).collect {
      case Tabbed(from, until, text) =>
        val f = MonthDay.parse(from, formatter)
        val u = MonthDay.parse(until, formatter)
        (f, u) -> text
    }.toIterable.toMultiMap
  }

  override def getReqAns: Iterator[(String, Pragmatic)] = {
    val aliases = for {
      (hotelId, names) <- getAliases.iterator
      name <- names
    } yield (name, HotelNameTop(hotelId))           // st.yandex-team.ru/HOTELS-2455#1507550757000

    val reqans = loadLines(DataTypes.wizardReqAns).collect {
      case Tabbed(request, IntValue(hotelId)) ⇒
        (request, HotelName(hotelId))
      case Tabbed(request, IntValue(hotelId), "1") ⇒
        (request, HotelNameTop(hotelId))
      case Tabbed(request, IntValue(hotelId), _) ⇒
        (request, HotelName(hotelId))
    }
    aliases ++ reqans
  }

  override def getAliases: collection.Map[Int, Seq[String]] = Map.empty
}
