package ru.yandex.tours.query.parser

import java.io.{File, OutputStream}
import java.nio.ByteBuffer
import java.util.concurrent.Callable

import com.google.common.cache.{Cache, CacheBuilder, Weigher}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.FileBaseHolder
import ru.yandex.tours.query.Pragmatic
import ru.yandex.tours.query.parser.ParsingTrie.Payload
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.io.SegmentedByteBuffer.SegmentedOutputStream
import ru.yandex.tours.util.io.{ByteBuffers, SegmentedByteBuffer}
import ru.yandex.tours.util.trie.Trie
import ru.yandex.tours.util.trie.mapped.MappedTrie

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.01.16
 */
class ParsingTrie(val trie: Trie, val parsingStates: ParsingStates,
                  private var cache: Cache[Integer, Array[ParsingTrie.Payload]] = null) {

  if (cache eq null) {
    cache = CacheBuilder.newBuilder()
      .concurrencyLevel(16)
      .maximumWeight(100000)
      .weigher(new Weigher[Integer, Array[ParsingTrie.Payload]] {
        def weigh(key: Integer, value: Array[ParsingTrie.Payload]): Int = value.length
      })
      .build[Integer, Array[ParsingTrie.Payload]]()
  }

  def getPragmatics(stateId: Int, useCache: Boolean = true): Array[ParsingTrie.Payload] = {
    if (!useCache) return parsingStates.getPragmatics(stateId)
    cache.get(stateId, new Callable[Array[ParsingTrie.Payload]] {
      override def call(): Array[Payload] = parsingStates.getPragmatics(stateId)
    })
  }

  // to escape from proxy wrapper
  def copy(): ParsingTrie = new ParsingTrie(trie, parsingStates, cache)

  override def toString: String =
    s"ParsingTrie($trie, $parsingStates)"

  def writeTo(os: OutputStream): Unit = {
    val segmented = new SegmentedOutputStream(os)

    segmented.writeSegment(ParsingTrie.TRIE_SEGMENT) { trie.writeTo }
    segmented.writeSegment(ParsingTrie.STATES_SEGMENT) { parsingStates.writeTo }

    segmented.close()
  }
}

object ParsingTrie extends Logging {
  case class Payload(length: Int, pragmatic: Pragmatic)

  private val TRIE_SEGMENT = "trie"
  private val STATES_SEGMENT = "states"

  def fromFile(file: File): ParsingTrie = {
    fromBuffer(ByteBuffers.mmap(file))
  }

  def fromBuffer(buffer: ByteBuffer): ParsingTrie = {
    val segmented = SegmentedByteBuffer(buffer)
    log.info("Loading ParsingTrie from " + segmented)
    val trieBuffer = segmented.getBuffer(TRIE_SEGMENT)
    val statesBuffer = segmented.getBuffer(STATES_SEGMENT)
    val trie = new MappedTrie(trieBuffer)
    val states = new MappedParsingStates(statesBuffer)
    new ParsingTrie(trie, states)
  }
}

class ParsingTrieDef(dir: File, dataType: DataType) extends FileBaseHolder[ParsingTrie](dir, dataType, false) {
  override protected def parseFile(file: File): ParsingTrie = {
    val buffer = ByteBuffers.mmap(file)
    ParsingTrie.fromBuffer(buffer)
  }
}