package ru.yandex.tours.wizard.microoffer

import ru.yandex.tours.model.wizard.MicroOffer.{HotelMicroOffersProto, MicroOfferProto}
import ru.yandex.tours.util.lang.RichOrdering
import ru.yandex.tours.wizard.resource.WizardResource

import scala.annotation.tailrec

/**
  * Created by asoboll on 31.01.17.
  */
trait MicroOfferIndex extends WizardResource {
  def get(hotelId: Int): Option[HotelMicroOffersProto]

  def iterator: Iterator[HotelMicroOffersProto]
}

object MicroOfferIndex {
  val offerPriceOrdering = Ordering.by((mo: MicroOfferProto) => mo.getPrice)

  val priceOrdering = Ordering.by((hotels: HotelMicroOffersProto) => hotels.minPrice)

  val countOrdering = Ordering.by((hotels: HotelMicroOffersProto) => hotels.getOffersCount <= 1)

  val defaultOrdering = countOrdering or priceOrdering

  val idOrdering = Ordering.by((hotels: HotelMicroOffersProto) => hotels.getHotelId)

  private implicit class RichHotelMicroOffersProto(hotelOffers: HotelMicroOffersProto) {
    def minPrice: Int = hotelOffers.getOffersList.get(0).getPrice
  }

  def distinctHotels(q: TraversableOnce[HotelMicroOffersProto]): Iterator[HotelMicroOffersProto] = {
    new Iterator[HotelMicroOffersProto] {
      private val it = q.toIterator.buffered

      override def hasNext: Boolean = it.hasNext

      @tailrec
      private def nextWithSameId(cur: HotelMicroOffersProto): HotelMicroOffersProto = {
        if (!(it.hasNext && (it.head.getHotelId == cur.getHotelId))) cur
        else nextWithSameId(List(cur, it.next).min(defaultOrdering))
      }

      override def next(): HotelMicroOffersProto = {
        if (it.hasNext) nextWithSameId(it.next)
        else throw new UnsupportedOperationException("Cannot call next on an exhausted iterator!")
      }
    }
  }
}
