package ru.yandex.tours.wizard.microoffer

import java.io.{File, FileInputStream}
import java.nio.ByteBuffer

import com.google.protobuf.ByteString
import org.joda.time.DateTime
import ru.yandex.tours.index.WizardIndexing
import ru.yandex.tours.model.wizard.MicroOffer.HotelMicroOffersProto
import ru.yandex.tours.util.collections.ByteBufferMappedMap.MappedMap
import ru.yandex.tours.util.io.ByteBuffers
import ru.yandex.tours.wizard.resource.{ExpiringWizardResource, WizardResourceLoader}

import scala.collection.immutable.TreeMap

/**
  * Created by asoboll on 01.02.17.
  */
class MicroOfferIndexImpl(val freshness: Long, map: Map[Int, HotelMicroOffersProto])
  extends MicroOfferIndex with ExpiringWizardResource {

  override def get(hotelId: Int): Option[HotelMicroOffersProto] = map.get(hotelId)

  override def iterator: Iterator[HotelMicroOffersProto] = map.valuesIterator

  def size: Int = map.size

  override def toString: String = s"MicroOfferIndex(${new DateTime(freshness)}, $size)"
}

object MicroOfferIndexImpl extends WizardResourceLoader[MicroOfferIndexImpl] {
  implicit val dK = (bb: ByteBuffer) => bb.getInt
  implicit val dV = (bb: ByteBuffer) => HotelMicroOffersProto.parseFrom(ByteString.copyFrom(bb))

  private def parseBuffer(buffer: ByteBuffer): (Long, Map[Int, HotelMicroOffersProto]) = {
    val formatVersion = buffer.getInt
    require(formatVersion == WizardIndexing.MicroOffers.formatVersion, s"Unknown format version: $formatVersion")
    val freshness = buffer.getLong
    val size = buffer.getInt
    val map = MappedMap.from[Int, HotelMicroOffersProto](buffer.slice)
    require(size == -1 || size == map.size, s"Record count ${map.size} doesnt match header: $size")
    (freshness, map)
  }

  def fromBuffer(buffer: ByteBuffer): MicroOfferIndexImpl = {
    val (freshness, map) = parseBuffer(buffer)
    new MicroOfferIndexImpl(freshness, map)
  }

  override def fromFile(file: File): MicroOfferIndexImpl = {
    val is = new FileInputStream(file)
    val buffer = ByteBuffers.mmap(file)
    val (freshness, map) = parseBuffer(buffer)
    new MicroOfferIndexImpl(freshness, map) {
      override def delete(): Unit = file.delete()
      override def load(): Unit = buffer.load()
      override def close(): Unit = ByteBuffers.unmap(buffer)
    }
  }

  override def empty: MicroOfferIndexImpl = new MicroOfferIndexImpl(0, Map.empty)

  def fromIterator(freshness: Long, hotels: Iterator[HotelMicroOffersProto]): MicroOfferIndexImpl = {
    val iterator = hotels.map(hotelOffers => hotelOffers.getHotelId -> hotelOffers)
    val map = TreeMap[Int, HotelMicroOffersProto](iterator.toSeq : _*)
    new MicroOfferIndexImpl(freshness, map)
  }
}
