package ru.yandex.tours.wizard.microoffer

import java.io._
import java.nio.ByteBuffer

import ru.yandex.tours.index.WizardIndexing
import ru.yandex.tours.model.wizard.MicroOffer.HotelMicroOffersProto
import ru.yandex.tours.util.collections.ByteBufferMappedMap.MappedMapWriter

/**
  * Created by asoboll on 13.02.17.
  */
trait MicroOfferWriter extends Closeable {
  def writeHeader(formatVersion: Int, freshness: Long, size: Int): Unit

  def writeItem(item: HotelMicroOffersProto): Unit

  def writeItems(items: TraversableOnce[HotelMicroOffersProto]): Unit
}

object MicroOfferWriter {
  import WizardIndexing.MicroOffers._

  def apply(os: OutputStream): MicroOfferWriter = new SingleMicroOfferWriter(os)

  def apply(file: File): MicroOfferWriter = {
    val os = new BufferedOutputStream(new FileOutputStream(file))
    apply(os)
  }

  def apply(writers: IndexedSeq[MicroOfferWriter]): MicroOfferWriter = new ShardedMicroOfferWriter(writers)

  private class SingleMicroOfferWriter(os: OutputStream) extends MicroOfferWriter {
    private def serializeInt(x: Int) = ByteBuffer.allocate(Integer.BYTES).putInt(x).array()
    private def serialize(item: HotelMicroOffersProto) = item.toByteArray
    private val mapWriter = new MappedMapWriter[Int, HotelMicroOffersProto](os)(serializeInt, serialize)

    def writeHeader(formatVersion: Int, freshness: Long, size: Int): Unit = {
      val header = ByteBuffer.allocate(headerSize)
        .putInt(formatVersion)
        .putLong(freshness)
        .putInt(size)
      os.write(header.array())
    }

    def writeItem(item: HotelMicroOffersProto): Unit = {
      mapWriter.write(item.getHotelId, item)
    }

    def writeItems(items: TraversableOnce[HotelMicroOffersProto]): Unit = {
      mapWriter.write(items.map(item => (item.getHotelId, item)))
    }

    def close(): Unit = {
      mapWriter.finish()
      os.close()
    }
  }

  private class ShardedMicroOfferWriter(writers: IndexedSeq[MicroOfferWriter]) extends MicroOfferWriter {
    require(writers.size == shardCount, s"Expected $shardCount writers, got ${writers.size}")

    def writeHeader(formatVersion: Int, freshness: Long, size: Int): Unit = {
      writers.foreach(_.writeHeader(formatVersion, freshness, -1))
    }

    def writeItem(item: HotelMicroOffersProto): Unit = {
      val i = shardId(item.getHotelId)
      writers(i).writeItem(item)
    }

    def writeItems(items: TraversableOnce[HotelMicroOffersProto]): Unit = {
      items.foreach(writeItem)
    }

    def close(): Unit = {
      writers.foreach(_.close())
    }
  }
}
