package ru.yandex.tours.wizard.resource

import java.io.{ByteArrayInputStream, File}
import java.util.zip.GZIPInputStream

import org.apache.commons.codec.digest.DigestUtils
import org.apache.commons.io.{FileUtils, IOUtils}
import ru.yandex.tours.util.GZip

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 12.03.15
 */
case class RawResource(bytes: Array[Byte], gzipped: Boolean, extension: String) {
  def size: Int = bytes.length
  def hash: String = DigestUtils.sha1Hex(bytes)
  override def toString: String = {
    val gz = if (gzipped) ", gzipped" else ""
    s"RawResource($hash, $size bytes$gz)"
  }

  def uniqueFileName: String = s"${System.currentTimeMillis()}_$hash$extension"

  def gzip: RawResource =
    if (gzipped) this
    else RawResource(GZip.compress(bytes), gzipped = true, extension)

  def writeTo(file: File): Unit = {
    val rawInputStream = new ByteArrayInputStream(bytes)
    val is =
      if (gzipped) new GZIPInputStream(rawInputStream)
      else rawInputStream
    val os = FileUtils.openOutputStream(file)

    try IOUtils.copy(is, os)
    finally IOUtils.closeQuietly(os)
  }
}
