package ru.yandex.tours

import akka.actor.Props
import akka.routing.RoundRobinPool
import ru.yandex.tours.app._
import ru.yandex.tours.extdata.{DataTypes, ExtDataSupport}
import ru.yandex.tours.query.parser.ParsingTrieDef
import ru.yandex.tours.util.LabelBuilder
import ru.yandex.tours.util.file.RichFile
import ru.yandex.tours.wizard.experiment.ExperimentControl
import ru.yandex.tours.wizard.geoaddr.GeoAddrPragmatics
import ru.yandex.tours.wizard.parser._
import ru.yandex.tours.wizard.query.PragmaticsParser
import ru.yandex.tours.wizard.search._
import ru.yandex.tours.wizard.serialize._
import ru.yandex.tours.wizard.{WizardHandler, WizardWarmer}

import scala.concurrent.duration._
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.01.15
 */
trait WizardComponents {
  app: Application with WizardIndexSupport
    with WizardMicroOffersSupport
    with AkkaSupport with SpraySupport
    with CommonDataHolders with ExtDataSupport
    with ZookeeperSupport
    with MetricsSupport =>

  val redirectUrl = Try(config.getString("tours.redirect.url")).getOrElse("")
  val labelBuilder = new LabelBuilder(config.getString("tours.label.key"))


  lazy val parser = {
    val trie = new ParsingTrieDef(app.dataFolder / "wizard_trie", DataTypes.wizardTrie)
      .from(extDataService, extDataUpdateChecker)
    new PragmaticsParser(trie)
  }

  lazy val reqAnsParser = {
    val trie = new ParsingTrieDef(app.dataFolder / "wizard_reqans_trie", DataTypes.wizardReqAnsTrie)
      .from(extDataService, extDataUpdateChecker)
    new PragmaticsParser(trie)
  }

  lazy val stopWordParser = {
    val trie = new ParsingTrieDef(app.dataFolder / "stop_word_trie", DataTypes.stopWordsTrie)
      .from(extDataService, extDataUpdateChecker)
    new PragmaticsParser(trie)
  }

  lazy val requestParser = {
    val geoAddPragmatics = new GeoAddrPragmatics(geoMappingHolder)
    val userRequestParser = new UserRequestParser(parser, reqAnsParser, stopWordParser,
      regionTree, hotelsIndex, hotelRatings, geoMappingHolder, directionsStats)
    val reporter = new InstrumentedBadQueryReporter(metricsRegistry)
    
    new WizardRequestParser(reporter, geoAddPragmatics, userRequestParser, departures, hotelRatings)
  }

  lazy val throttler = new ZkWizardThrottler(zkClient)
  lazy val experimentControl = new ExperimentControl(zkClient)

  lazy val freshIndexSearcher = new IndexToursSearcher(freshIndex, hotelsIndex, hotelRatings,
    regionTree, directions, hotDirections, false) with DirectionCaching with InstrumentedToursSearcher {
    override def metrics = metricsRegistry
    override def marker = "fresh"
  }
  lazy val longIndexSearcher = new IndexToursSearcher(longIndex, hotelsIndex, hotelRatings,
    regionTree, directions, hotDirections, true) with DirectionCaching with InstrumentedToursSearcher {
    override def metrics = metricsRegistry
    override def marker = "long"
  }
  lazy val roomsSearcher = new RoomsSearcher(hotelsIndex, hotelRatings,
    regionTree, geoMappingHolder, directions, directionsStats)(akkaSystem.dispatcher) with InstrumentedToursSearcher {
    override def metrics = metricsRegistry
    override def marker = "rooms"
  }
  lazy val freshOffersSearcher = new MicroOffersSearcher(freshOffers) with InstrumentedMicroOffersSearcher {
    override def metrics = metricsRegistry
    override def marker = "offers_fresh"
  }
  lazy val longOffersSearcher = new MicroOffersSearcher(longOffers) with InstrumentedMicroOffersSearcher {
    override def metrics = metricsRegistry
    override def marker = "offers_long"
  }
  lazy val searcher = new LoggingToursSearcher(
    new AddOffersToursSearcher(
      new SequenceToursSearcher(
        new FallbackDepartureToursSearcher(
          new SequenceToursSearcher(
            freshIndexSearcher,
            longIndexSearcher
          )
        ),
        roomsSearcher
      ),
      freshOffersSearcher,
      longOffersSearcher
    )
  )

  lazy val textBuilder = new TextBuilder(regionTree, geoMappingHolder, searchSettings, translations, tourOperators, hotelProviders)

  lazy val serializer = {
    val reportSerializer = new ReportProtobufSerializer
    reportSerializer.setUrl("https://" + frontHost + "/?utm_source=wizard")
    reportSerializer.setSource("tours")
    reportSerializer.setGrouping("tours")
    reportSerializer.setEncoding("UTF-8")
    val jsonBuilder = new DefaultJsonBuilder(regionTree, geoMappingHolder, searchSettings, textBuilder, urlBuilder,
      redirectUrl, labelBuilder)

    new WizardResponseSerializer(reportSerializer, jsonBuilder)
  }

  lazy val wizardWarmer = new WizardWarmer(hotelsIndex, geoMappingHolder, freshIndex, longIndex)

  lazy val handlerProps = {
    Props(new WizardHandler(metricsRegistry, requestParser, throttler, experimentControl, searcher, serializer))
      .withRouter(RoundRobinPool(nrOfInstances = 16, routerDispatcher = "akka.actor.handler-dispatcher"))
      .withDispatcher("akka.actor.handler-dispatcher")
  }

  onStart {
    geoMappingHolder
    regionTree
    hotelsIndex
    directions
    translations
    departures
    requestParser

    roomsSearcher.warmUp()
    startServer(handlerProps, config.getInt("component.port"))
    wizardWarmer.schedule(10.minutes)(akkaSystem)
  }

  onStop {
    throttler.close()
  }
}
