package ru.yandex.tours

import java.io.File

import akka.actor.Props
import ru.yandex.common.pusher.PusherExtension
import ru.yandex.tours.app.{AkkaSupport, Application, ZookeeperSupport}
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.index.shard.IndexShard
import ru.yandex.tours.index.{IndexCleaner, WizardIndexing}
import ru.yandex.tours.util.file._
import ru.yandex.tours.wizard.index.{FreshIndexHolder, LongIndexHolder}
import ru.yandex.tours.wizard.resource.ResourceReceiver

import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 12.03.15
 */
trait WizardIndexSupport { app: Application with AkkaSupport with ZookeeperSupport with ExtDataSupport =>

  private val indexDir = new File(app.config.getString("tours.wizard.index.dir"))

  private val freshDir = indexDir / "fresh"
  private val longDir = indexDir / "long"

  private val freshTTL = 20.minutes
  private val longTTL = 24.hours

  lazy val freshIndex = FreshIndexHolder.fromDirectory("fresh", freshDir, freshTTL)
  lazy val longIndex = LongIndexHolder("long", longDir, extDataService, extDataUpdateChecker)

  lazy val indexReceiver = akkaSystem.actorOf(Props(
    new ResourceReceiver(freshIndex, IndexShard, freshDir, "index")))

  onStart {
    PusherExtension(akkaSystem)
      .newSubscriber(WizardIndexing.topic, indexReceiver, app.dataCenter, zkClient, WizardIndexing.zkPath)

    IndexCleaner.start(freshDir, WizardIndexing.indexExtension, freshTTL, 1.hour)(akkaSystem)
    IndexCleaner.start(longDir, WizardIndexing.indexExtension, longTTL, 1.hour)(akkaSystem)

    freshIndex.startMonitoring(isCritical = false)(akkaSystem)
    longIndex.startMonitoring(isCritical = true)(akkaSystem)
  }
}
