package ru.yandex.tours.wizard

import com.codahale.metrics.MetricRegistry
import org.slf4j.LoggerFactory
import ru.yandex.tours.util.Logging
import ru.yandex.tours.wizard.experiment.{ExperimentControl, WizardExperiment}
import ru.yandex.tours.wizard.parser.WizardRequestParser
import ru.yandex.tours.wizard.query.WizardRequest
import ru.yandex.tours.wizard.search.{WizardThrottler, WizardToursSearcher}
import ru.yandex.tours.wizard.serialize.WizardResponseSerializer
import spray.http.StatusCodes
import spray.routing.{HttpServiceActor, Route}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.01.15
 */
class WizardHandler(metrics: MetricRegistry,
                    parser: WizardRequestParser,
                    throttler: WizardThrottler,
                    experimentControl: ExperimentControl,
                    searcher: WizardToursSearcher,
                    serializer: WizardResponseSerializer) extends HttpServiceActor with Logging {

  private val timeoutLogger = LoggerFactory.getLogger("wizard.timeout")
  private val timeoutMs = 45

  private val wizardTimer = metrics.timer("wizard.time")
  private val timeoutMeter = metrics.meter("wizard.timeout")
  private val timeoutWithDataMeter = metrics.meter("wizard.timeout_data")

  private val supportedDomains = Set("ru")

  private def userRequest = parameter('user_request) | parameter('text)

  lazy val route: Route = {
    path("wizard") {
      (userRequest & parameters('geoaddr, 'region_id.as[Int], 'human_readable.as[Boolean] ? false, 'tld, 'uid ? "")) {
        (userRequest, geoaddr, regionId, humanReadable, tld, uid) =>
          WizardTracer.init()
          WizardExperiment.update(uid)
          val timer = wizardTimer.time()
          val wizardRequest = WizardRequest(regionId, userRequest, geoaddr)

          if (supportedDomains.contains(tld) && throttler.shouldAnswer(userRequest, uid)) {
            val data = for {
              request <- parser.parse(wizardRequest)
              response <- searcher.search(request)
            } yield {
              WizardTracer.checkpoint("start_serialize")
              val res = serializer.serialize(
                request,
                response,
                humanReadable
              )
              WizardExperiment.logResponse(response.wizardType)
              res
            }
            WizardTracer.checkpoint("finish")
            val handlingTime = timer.stop() / 1000000
            if (handlingTime > timeoutMs) {
              timeoutMeter.mark()
              if (data.isDefined) timeoutWithDataMeter.mark()
              timeoutLogger.info(s"TIMEOUT\thandling_time=$handlingTime\ttimeout=$timeoutMs\tuser_request=$userRequest\tregion_id=$regionId\t${WizardTracer.stats}")
            }

            WizardTracer.clear()
            WizardExperiment.clear()
            data match {
              case Some(bytes) => complete(StatusCodes.OK, bytes)
              case None => complete(StatusCodes.NoContent, "")
            }
          } else {
            //throttled
            WizardTracer.clear()
            WizardExperiment.clear()
            complete(StatusCodes.NoContent, "")
          }
      }
    } ~ (path("throttler") & parameter('probability.as[Double])) { prob =>
      throttler.setProbability(prob)
      complete(StatusCodes.OK, "")
    } ~ (path("experiment") & parameters('name, 'enabled.as[Boolean])) { (name, enabled) =>
      if (enabled) experimentControl.enable(name)
      else experimentControl.disable(name)
      complete(StatusCodes.OK, "")
    }
  }

  override def receive: Receive = runRoute(route)
}
