package ru.yandex.tours.wizard

import it.unimi.dsi.fastutil.ints.IntArrayList
import it.unimi.dsi.fastutil.longs.LongArrayList
import ru.yandex.tours.util.parsing.TSKV

import scala.collection.mutable.ArrayBuffer

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 09.04.15
 */
object WizardTracer {
  private val INITIAL_CAPACITY = 18

  private class Trace {
    val started = System.nanoTime()
    var last = started
    var i = 0
    val names = new ArrayBuffer[String](INITIAL_CAPACITY)
    val times = new LongArrayList(INITIAL_CAPACITY)
    val items = new IntArrayList(INITIAL_CAPACITY)

    def update(name: String, count: Int = -1): Unit = {
      val now = System.nanoTime()
      names += name
      times.add(now - last)
      items.add(count)
      last = now
      i += 1
    }
  }

  private val local = new ThreadLocal[Trace]

  def init(): Unit = local.set(new Trace)
  def clear(): Unit = local.set(null)

  def checkpoint(name: String, count: Int = -1): Unit = {
    val trace = local.get()
    if (trace ne null) {
      trace.update(name, count)
    }
  }

  def stats: String = {
    val trace = local.get()
    if (trace ne null) {
      TSKV(
        trace.names.zipWithIndex.map { case (name, i) =>
          val time = trace.times.get(i)
          val items = trace.items.get(i)
          val value = (time / 1000000) + "ms" + (if (items >= 0) "/" + items.toString else "")
          name -> value
        }
      )
    } else {
      ""
    }
  }
}
