package ru.yandex.tours.wizard.domain

import ru.yandex.tours.model.filter.hotel.HotelTypeFilter
import ru.yandex.tours.query._

import scala.reflect._

/**
 * WizardRequest mapped to our domain model
 *
 * @param originalRequest original user_request
 * @param markers marker (e.g. туры, отдых, отель, дешевый)
 * @param when date interval of tour start (e.g. в сентябре, на майские)
 * @param from departure location (e.g. из Петербурга)
 * @param to destination location (country/city) (e.g. в Египет, в Мадрид)
 * @param hotel hotel (e.g. Caprice Beach hotel)
 * @param operator tour operator (e.g. Пегас, тез тур)
 */
case class ToursWizardRequest(originalRequest: String,
                              userRegion: Int,
                              regionInRequest: Option[Int],
                              markers: Set[Marker],
                              when: TourStartInterval,
                              from: Int,
                              fallbackFrom: Seq[Int],
                              to: Option[Int],
                              hotel: Option[Int],
                              operator: Option[Int],
                              hotelOnTop: Boolean = false) {

  def hasMarkerOf[T <: Marker : ClassTag]: Boolean =
    markers.exists(classTag[T].runtimeClass.isInstance)

  def collectOf[T <: Marker : ClassTag]: Set[T] = markers.collect {
    case m if classTag[T].runtimeClass.isInstance(m) => m.asInstanceOf[T]
  }

  def hasMarker: Boolean = markers.contains(TourMarker) || markers.contains(SearchMarker) ||
    markers.contains(PriceMarker) || markers.contains(BuyMarker)

  def hasCheapMarker: Boolean = markers.contains(CheapMarker)
  def hasSkiMarker: Boolean = markers.contains(SkiResorts)
  def hasHotelMarker: Boolean = hasMarkerOf[HotelMarker]
  def hasOfficialSiteMarker: Boolean = markers.contains(OfficialSiteMarker)
  def hasHotMarker: Boolean = markers.contains(HotMarker)

  def hotelTypes: Set[QueryHotelType] = collectOf[HotelMarker].map(_.hotelType)
  def hotelTypesFilter: HotelTypeFilter = HotelTypeFilter(hotelTypes.flatMap(_.getSearchTypes))
}