package ru.yandex.tours.wizard.domain

import org.joda.time.LocalDate
import ru.yandex.tours.direction.Direction
import ru.yandex.tours.model.BaseModel.Pansion
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.wizard.MicroOffer.HotelMicroOffersProto
import ru.yandex.tours.resorts.Ski
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.wizard.WizardType

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.01.15
 */
sealed trait ToursWizardResponse {
  def from: Int
  def wizardType: WizardType
}

// responses
case class RoomsResponse(from: Int, to: Int, results: Seq[Hotel]) extends ToursWizardResponse {
  override val wizardType: WizardType = WizardType("tours/hotels", "rooms", counter = "tours/rooms/hotels")
}

case class ToursResponse(from: Int, to: Int, results: Seq[TourWithPrice]) extends ToursWizardResponse {
  override val wizardType: WizardType = WizardType("tours/hotels", "tours")
}

case class SkiResortResponse(from: Int, to: Int, ski: Ski, results: Seq[Hotel]) extends ToursWizardResponse {
  override val wizardType: WizardType = WizardType("tours/hotels", "ski", counter = "tours/ski/hotels")
}

case class RoomDirectionsResponse(from: Int, directions: Seq[Direction]) extends ToursWizardResponse {
  override val wizardType: WizardType = WizardType("tours/directions", "rooms", counter = "tours/rooms/directions")
}

case class TourDirectionsResponse(from: Int, directions: Seq[DirectionWithPrice]) extends ToursWizardResponse {
  override val wizardType: WizardType = WizardType("tours/directions", "tours")
}

case class HotelPricesResponse(from: Int, hotel: Hotel, prices: Seq[HotelPrice],
                               offers: Option[HotelMicroOffersProto] = None) extends ToursWizardResponse {
  override def wizardType: WizardType = WizardType("tours/hotel/prices", "tours")
}

case class HotelSnippetResponse(from: Int, bestPrice: Option[HotelPrice], hotel: Hotel,
                                offers: Option[HotelMicroOffersProto] = None) extends ToursWizardResponse {
  private def isToursHotel = hotel.toursSearchAvailable
  override def wizardType: WizardType = WizardType("tours/hotel/snippet",
    context = if (isToursHotel) "tours" else "rooms",
    counter = if (isToursHotel) "tours/hotel/snippet" else "tours/rooms/hotel/snippet"
  )
}

// containers
case class TourWithPrice(whenInt: Int, nights: Int, hotel: Hotel, minPrice: Int, pansion: Pansion) {
  def when: LocalDate = whenInt.toLocalDate
}

case class DirectionWithPrice(whenInt: Int, nights: Int, direction: Direction, minPrice: Int) {
  def when: LocalDate = whenInt.toLocalDate
}

case class HotelPrice(whenInt: Int, nights: Int, minPrice: Int) {
  def when: LocalDate = whenInt.toLocalDate
}

object Orderings {
  implicit val directionWithPrice: Ordering[DirectionWithPrice] =
    Ordering.by((e: DirectionWithPrice) => (e.direction.maxRelevance, -e.minPrice))

  implicit val hotelPrice: Ordering[HotelPrice] =
    Ordering.by((e: HotelPrice) => -e.minPrice)
}
