package ru.yandex.tours.wizard.geoaddr

import org.apache.commons.codec.Charsets
import org.apache.commons.codec.binary.Base64
import org.json.{JSONObject, JSONArray}
import ru.yandex.tours.geo.base.region

import scala.collection.JavaConverters._
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.01.15
 */
object GeoAddrParser {

  def fromBase64(content: String): GeoAddr = {
    if (content.nonEmpty) {
      val json = new String(Base64.decodeBase64(content), Charsets.UTF_8)
      fromJson(json)
    } else {
      GeoAddr.empty
    }
  }

  def fromJson(jsonString: String): GeoAddr = {
    val array = new JSONArray(jsonString)
    val objects =
      for {
        i <- 0 until array.length()
        parsed <- Try { parseAddrObject(array.getJSONObject(i)) }.toOption
      } yield parsed

    GeoAddr(objects: _*)
  }

  def parseAddrObject(json: JSONObject): GeoAddrObject = {
    val pos = json.getInt("pos")
    val length = json.getInt("length")
    val name = Option(json.optString("geo", null)).orElse(Option(json.optString("addr", null))).getOrElse("")
    val normalized = json.getString("normalizedtext")
    val bestGeo = json.optInt("bestgeo")

    val data = json.getJSONObject("data")
    val variantsArray = data.getJSONArray("Variants")
    val variants =
      for (i <- 0 until variantsArray.length())
      yield parseVariant(variantsArray.getJSONObject(i))

    GeoAddrObject(variants, bestGeo, name, normalized, pos, length)
  }

  def parseVariant(json: JSONObject): GeoVariant = {
    val weight = json.getDouble("Weight")
    val keys = json.keys().asScala.toList.asInstanceOf[List[String]]
    val geoIds = for {
      name <- keys
      if name.endsWith("IDs")
      array = json.getJSONArray(name)
      i <- 0 until array.length()
    } yield array.getInt(i)

    val names = region.Types.values.map(_.toString)
    val typeString = keys.find(names.contains)
    val `type` = typeString.fold(region.Types.Other)(region.Types.withName)

    GeoVariant(geoIds.distinct, `type`, weight)
  }
}
